/* CityManager.m
 * controller class and files owner of the CityPanel
 *
 * Copyright (C) 2003-2004 by vhf interservice GmbH
 * Author:   Georg Fleischmann
 *
 * created:  2003-06-29
 * modified: 2004-10-06
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 *
 * vhf interservice GmbH, Im Marxle 3, 72119 Altingen, Germany
 * eMail: info@vhf.de
 * http://www.vhf.de
 */

#include "CityManager.h"
#include "../Cenon/VHFShared/VHFDictionaryAdditions.h"
#include "../Cenon/App.h"
#include "../Cenon/functions.h"
#include "../Cenon/messages.h"
#include "astroMessages.h"
#include "AstroPrincipal.h"
#include "Astro.bproj/AstroController.h"

#define CITIES_FOLDER @"Cities"

@interface CityManager(PrivateMethods)
- (NSMutableDictionary*)cityDictionaryWithName:(NSString*)name andString:(NSString*)cityString;
- (NSMutableString*)cityStringWithDictionary:(NSDictionary*)city;
- (void)loadStates;
- (void)loadCitiesOfState:(NSString*)name;
- (void)saveCitiesOfState:(NSString*)name;
@end

@implementation CityManager

/* created: 2003-06-29
 * sort city dictionaries by name
 */
/*static int sortCityDictionary(id event1, id event2, void *context)
{
    return [(NSString*)[event1 objectForKey:@"city"] compare:[event2 objectForKey:@"city"]];
}*/

/* 
 */
- (id)init
{
#if 0	// convert state list
    {   NSDictionary	*dict = [NSDictionary dictionaryWithContentsOfFile:@"/tmp/state.dict"];
        NSMutableArray	*vals = [[[dict allValues] mutableCopy] autorelease];
        int		i;

        for (i=0; i<[vals count]; i++)
            if ([[vals objectAtIndex:i] isKindOfClass:[NSArray class]])
                [vals replaceObjectAtIndex:i withObject:[[vals objectAtIndex:i] objectAtIndex:0]];

        [vals sortUsingSelector:@selector(compare:)];
        dict = [NSDictionary dictionaryWithObjects:vals forKeys:vals];
        [dict writeToFile:@"/tmp/States.strings" atomically:YES];
    }
#endif
    [super init];

    return self;
}

/* load interface file and display panel
 */
- (void)showPanel:sender
{
    if (!cityPanel)
    {   NSBundle	*bundle = [NSBundle bundleForClass:[CityManager class]];

        /* load city panel */
        if ( ![bundle loadNibFile:@"CityPanel"
                externalNameTable:[NSDictionary dictionaryWithObject:self forKey:@"NSOwner"]
                         withZone:[self zone]] )
            NSLog(@"Cannot load CityPanel interface file");
        [cityPanel setFrameUsingName:@"CityPanel"];
        [cityPanel setFrameAutosaveName:@"CityPanel"];

        [self loadStates];
        [cityBrowser loadColumnZero];
        [self clear:self];
    }
    [cityPanel makeKeyAndOrderFront:sender];
}


/*
 * managing event folders
 */

#if 0
- (void)addFolder:sender;
{   NSString	*name = [folderField stringValue];

    if (![folders containsObject:name])
    {   [folders addObject:name];
        [folders sortUsingSelector:@selector(compare:)];

        [cities release];
        cities = [NSMutableDictionary dictionary];
        [self saveCitiesOfState:name];

        [cityBrowser loadColumnZero];
        [cityBrowser selectRow:[folders count]-1 inColumn:0];
    }
    else
        NSLog(@"CityManager, Add Folder: Name '%@' already in use!", name);
}
- (void)renameFolder:sender;
{   NSFileManager	*fileManager = [NSFileManager defaultManager];
    int			folderIx = [cityBrowser selectedRowInColumn:0];
    NSString		*name = [folderField stringValue];

    if (name)
    {
        if (![folders containsObject:name])
        {
            srcPath = [NSString stringWithFormat:@"%@/Astro/%@/%@", userLibrary(), CITIES_FOLDER, [folders objectAtIndex:folderIx]];
            dstPath = [NSString stringWithFormat:@"%@/Astro/%@/%@", userLibrary(), CITIES_FOLDER, name];
            [fileManager movePath:srcPath toPath:dstPath handler:NULL];
            [folders replaceObjectAtIndex:folderIx withObject:name];
            [cityBrowser loadColumnZero];
            [cityBrowser selectRow:folderIx inColumn:0];
        }
        else
            NSLog(@"CityManager, Rename Folder: Name '%@' already in use", name);
    }
    else
        NSLog(@"CityManager, Rename Folder: Name == nil");
}
- (void)removeFolder:sender;
{   int	folderIx = [cityBrowser selectedRowInColumn:0];

    if (! NSRunAlertPanel(@"", REALLYDELETEFOLDER_STRING, DELETE_STRING, CANCEL_STRING, nil,
                               [folders objectAtIndex:folderIx]) == NSAlertDefaultReturn)
        return;

    [folders removeObjectAtIndex:folderIx];
    [cityBrowser loadColumnZero];
    [cityBrowser selectRow:(folderIx >= 1) ? folderIx-1 : 0 inColumn:0];
    [self saveCities];
}
#endif

/*
 * managing cities
 */

- (void)add:sender
{   NSMutableDictionary	*city = [NSMutableDictionary dictionary];
    int			folderIx = [cityBrowser selectedRowInColumn:0], cityIx;

    if (folderIx < 0)
        return;

    /* add data fields */
    if (![cityField stringValue] || [cityKeys containsObject:[cityField stringValue]])
    {   NSLog(@"CityManager: City '%@' already in database", [cityField stringValue]);
        return;
    }
    [city setObject:[cityField stringValue]    forKey:@"city"];
    [city setObject:[countryField stringValue] forKey:@"country"];
    [city setObject:[zipField stringValue]     forKey:@"zip"];
    [city setObject:[zipMaxField stringValue]  forKey:@"zipMax"];
    [city setObject:[latField stringValue]     forKey:@"lat"];
    [city setObject:[lonField stringValue]     forKey:@"lon"];
    [city setObject:[elevField stringValue]    forKey:@"elev"];
    [cities setObject:[self cityStringWithDictionary:city] forKey:[city objectForKey:@"city"]];

    [cityKeys addObject:[city objectForKey:@"city"]];
    [cityKeys sortUsingSelector:@selector(compare:)];
    cityIx = [cityKeys indexOfObject:[city objectForKey:@"city"]];

    [cityBrowser reloadColumn:1];
    [cityBrowser selectRow:cityIx inColumn:1];

    [self saveCitiesOfState:[folders objectAtIndex:folderIx]];
}
- (void)modify:sender
{   int			folderIx = [cityBrowser selectedRowInColumn:0];
    int			cityIx  = [cityBrowser selectedRowInColumn:1];
    NSString		*name;
    NSMutableDictionary	*city;

    if (cityIx < 0)
        return;

    name = [cityKeys objectAtIndex:cityIx];
    city = [self cityDictionaryWithName:name andString:[cities objectForKey:name]];
    if (! NSRunAlertPanel(@"", REALLYMODIFY_STRING, MODIFY_STRING, CANCEL_STRING, nil,
                               name) == NSAlertDefaultReturn)
        return;

    /* update values */
    if ( ![[cityField stringValue] isEqual:[city objectForKey:@"city"]]
         && [[cityField stringValue] length]
         && ![cityKeys containsObject:[cityField stringValue]] )
    {   [city setObject:[cityField stringValue] forKey:@"city"];
        [cityKeys replaceObjectAtIndex:cityIx withObject:[cityField stringValue]];
        [cities removeObjectForKey:name];
        [cityKeys sortUsingSelector:@selector(compare:)];
    }
    [city setObject:[zipField stringValue]     forKey:@"zip"];
    [city setObject:[zipMaxField stringValue]  forKey:@"zipMax"];
    [city setObject:[countryField stringValue] forKey:@"country"];
    [city setObject:[latField stringValue]     forKey:@"lat"];
    [city setObject:[lonField stringValue]     forKey:@"lon"];
    [city setObject:[elevField stringValue]    forKey:@"elev"];
    [cities setObject:[self cityStringWithDictionary:city] forKey:[city objectForKey:@"city"]];

    [cityBrowser reloadColumn:1];
    [cityBrowser selectRow:cityIx inColumn:1];

    [self saveCitiesOfState:[folders objectAtIndex:folderIx]];
}
- (void)remove:sender
{   int			folderIx = [cityBrowser selectedRowInColumn:0];
    int			cityIx  = [cityBrowser selectedRowInColumn:1];
    NSString		*name;

    if (cityIx < 0)
        return;

    name = [cityKeys objectAtIndex:cityIx];
    if (! NSRunAlertPanel(@"", REALLYDELETECITY_STRING, DELETE_STRING, CANCEL_STRING, nil,
                               name) == NSAlertDefaultReturn)
        return;

    [cities removeObjectForKey:name];
    [cityKeys removeObjectAtIndex:cityIx];
    [self clear:self];
    [cityBrowser reloadColumn:1];

    [self saveCitiesOfState:[folders objectAtIndex:folderIx]];
}

- (void)clear:sender
{
    [cityField    setStringValue:@""];
    [countryField setStringValue:@""];
    [zipField     setStringValue:@""];
    [zipMaxField  setStringValue:@""];
    [latField     setStringValue:@""];
    [lonField     setStringValue:@""];
    [elevField    setStringValue:@""];
    [cityBrowser selectRow:[cityBrowser selectedRowInColumn:0] inColumn:0];
}

/* search in current city list
 */
- (void)search:sender
{   int			i, cityIx = -1;
    NSString		*string, *name;
    NSDictionary	*city;

    if ([string=[cityField stringValue] length])
    {
        for (i=0; i<[cityKeys count]; i++)
            if ([[cityKeys objectAtIndex:i] rangeOfString:string].length)
            {   cityIx = i;
                break;
            }
    }

    if (cityIx < 0)
        return;
    [cityBrowser selectRow:cityIx inColumn:1];

    name = [cityKeys objectAtIndex:cityIx];
    city = [self cityDictionaryWithName:name andString:[cities objectForKey:name]];
    [cityField    setStringValue:[city objectForKey:@"city"]];
    [countryField setStringValue:(string=[city objectForKey:@"country"]) ? string : @""];
    [zipField     setStringValue:(string=[city objectForKey:@"zip"])  ? string : @""];
    [zipMaxField  setStringValue:(string=[city objectForKey:@"zipMax"])  ? string : @""];
    [latField     setStringValue:(string=[city stringForKey:@"lat"])  ? string : @""];
    [lonField     setStringValue:(string=[city stringForKey:@"lon"])  ? string : @""];
    [elevField    setStringValue:(string=[city stringForKey:@"elev"]) ? string : @""];
}

/* set selection in EventPanel etc. via notification
 */
- (void)set:sender
{   int			cityIx   = [cityBrowser selectedRowInColumn:1];
    NSString		*name;
    NSDictionary	*city;

    if (cityIx < 0)
        return;

    name = [cityKeys objectAtIndex:cityIx];
    city = [self cityDictionaryWithName:name andString:[cities objectForKey:name]];
    [[NSNotificationCenter defaultCenter] postNotificationName:AstroUpdateCity
                                                        object:city userInfo:nil];
}


/*
 * internal methods
 */

- (void)loadStates
{   NSString		*path;
    NSFileManager	*fileManager = [NSFileManager defaultManager];

    [cityBrowser setDelegate:self];
    [cityBrowser setTarget:self];
    [cityBrowser setAction:@selector(doClick:)];

    [folders release];
    folders = [NSMutableArray new];

    path = [NSString stringWithFormat:@"%@/Astro/%@", userLibrary(), CITIES_FOLDER];
    if ([fileManager fileExistsAtPath:path])
        [folders addObjectsFromArray:[fileManager directoryContentsAtPath:path]];

    path = [NSString stringWithFormat:@"%@/Astro/%@", localLibrary(), CITIES_FOLDER];
    if ([fileManager fileExistsAtPath:path])
        [folders addObjectsFromArray:[fileManager directoryContentsAtPath:path]];

    //folders = [[fileManager directoryContentsAtPath:path] mutableCopy];
    [folders sortUsingSelector:@selector(compare:)];
}

/* load cities from file
 */
- (void)loadCitiesOfState:(NSString*)name
{   NSString		*path;
    NSFileManager	*fileManager = [NSFileManager defaultManager];

    path = [NSString stringWithFormat:@"%@/Astro/%@/%@", userLibrary(), CITIES_FOLDER, name];
    if ( ![fileManager fileExistsAtPath:path] )
        path = [NSString stringWithFormat:@"%@/Astro/%@/%@", localLibrary(), CITIES_FOLDER, name];

    [cities release];
    cities = [[NSMutableDictionary dictionaryWithContentsOfFile:path] retain];
    [cityKeys release];
    cityKeys = [[cities allKeys] mutableCopy];
    [cityKeys sortUsingSelector:@selector(compare:)];
}
- (void)saveCitiesOfState:(NSString*)name
{   NSString		*path, *dir;
    NSFileManager	*fileManager = [NSFileManager defaultManager];

    dir = [userLibrary() stringByAppendingPathComponent:@"Astro"];
    if (![fileManager fileExistsAtPath:dir])	// create Astro folder
        [fileManager createDirectoryAtPath:dir attributes:nil];
    dir = [dir stringByAppendingPathComponent:CITIES_FOLDER];
    if (![fileManager fileExistsAtPath:dir])	// create city folder
        [fileManager createDirectoryAtPath:dir attributes:nil];

    path = [dir stringByAppendingPathComponent:name];
    [cities writeToFile:path atomically:YES];
}

/* expand city data from string into dictionary
 */
- (NSMutableDictionary*)cityDictionaryWithName:(NSString*)name andString:(NSString*)cityString
{   NSMutableDictionary	*city = [NSMutableDictionary dictionary];
    NSScanner		*scanner = [NSScanner scannerWithString:cityString];
    NSString		*string;
    float		f;
    int			i;

    [city setObject:name forKey:@"city"];
    if ([scanner scanUpToString:@" " intoString:&string])
        [city setObject:string forKey:@"country"];
    if ([scanner scanFloat:&f])
        [city setObject:vhfStringWithFloat(f) forKey:@"lat"];
    if ([scanner scanFloat:&f])
        [city setObject:vhfStringWithFloat(f) forKey:@"lon"];
    if ([scanner scanInt:&i])
        [city setObject:[NSNumber numberWithInt:i] forKey:@"elev"];
    if ([scanner scanUpToString:@" " intoString:&string])
        [city setObject:string forKey:@"zip"];
    if ([scanner scanUpToString:@" " intoString:&string])
        [city setObject:string forKey:@"zipMax"];
    return city;
}
/* compress city data from dictionary into string
 */
- (NSMutableString*)cityStringWithDictionary:(NSDictionary*)city
{   NSMutableString	*cityString = [NSMutableString string];
    NSString		*string;

    [cityString appendString:([string=[city objectForKey:@"country"] length]) ? string : @"--"];
    [cityString appendFormat:@" %@", ([string=[city objectForKey:@"lat"]  length]) ? string : @"0.0"];
    [cityString appendFormat:@" %@", ([string=[city objectForKey:@"lon"]  length]) ? string : @"0.0"];
    [cityString appendFormat:@" %@", ([string=[city objectForKey:@"elev"] length]) ? string : @"0"];
    [cityString appendFormat:@" %@", ([string=[city objectForKey:@"zip"]  length]) ? string : @"0"];
    if ((string=[city objectForKey:@"zipMax"]))
        [cityString appendFormat:@" %@", string];
    return cityString;
}

/* delegate methods
 */
- (int)browser:(NSBrowser*)sender numberOfRowsInColumn:(int)column
{
    switch (column)
    {
        case 0: return [folders count];
        case 1: return [cityKeys count];
    }
    return 0;
}

-(void)browser:(NSBrowser*)sender willDisplayCell:(id)cell atRow:(int)row column:(int)column
{   NSString	*name;

    switch (column)
    {
        case 0:
            name = [folders objectAtIndex:row];
            name = [[NSBundle bundleForClass:[self class]] localizedStringForKey:name value:nil table:@"States"];
            [cell setStringValue:name];
            [cell setLeaf:NO];
            break;
        case 1:
            [cell setStringValue:[cityKeys objectAtIndex:row]];
            [cell setLeaf:YES];
            break;
    }

    [cell setLoaded:YES];
}

- (void)doClick:sender
{   int			folderIx = [cityBrowser selectedRowInColumn:0];
    int			cityIx  = [cityBrowser selectedRowInColumn:1];
    NSDictionary	*city;
    NSString		*name, *string;

    if (cityIx < 0)
    {
        name = [folders objectAtIndex:folderIx];
        name = [[NSBundle bundleForClass:[self class]] localizedStringForKey:name value:nil
                                                                       table:@"States"];
        [folderField setStringValue:name];
        [self loadCitiesOfState:[folders objectAtIndex:folderIx]];
        [cityBrowser reloadColumn:1];
        [self clear:self];
        return;
    }

    name = [cityKeys objectAtIndex:cityIx];
    city = [self cityDictionaryWithName:name andString:[cities objectForKey:name]];
    [cityField    setStringValue:[city objectForKey:@"city"]];
    [countryField setStringValue:(string=[city objectForKey:@"country"]) ? string : @""];
    [zipField     setStringValue:(string=[city objectForKey:@"zip"])  ? string : @""];
    [zipMaxField  setStringValue:(string=[city objectForKey:@"zipMax"])  ? string : @""];
    [latField     setStringValue:(string=[city stringForKey:@"lat"])  ? string : @""];
    [lonField     setStringValue:(string=[city stringForKey:@"lon"])  ? string : @""];
    [elevField    setStringValue:(string=[city stringForKey:@"elev"]) ? string : @""];
}

- (void)dealloc
{
    [cities release];
    [cityKeys release];
    [folders release];
    [super dealloc];
}

@end
