/* Waves.m
 * This calculates waves
 *
 * Copyright (C) 2005 by vhf interservice GmbH
 * Author:   Georg Fleischmann
 *
 * created:  2005-07-27
 * modified: 2005-07-28
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 *
 * vhf interservice GmbH, Im Marxle 3, 72119 Altingen, Germany
 * eMail: info@vhf.de
 * http://www.vhf.de
 */

#include <AppKit/AppKit.h>
//#include <VHFShared/VHFStringAdditions.h>
#include <VHFShared/VHFArrayAdditions.h>
#include <VHFShared/VHFDictionaryAdditions.h>
//#include "../Cenon/functions.h"
#include "../Cenon/Graphics.h"
#include "Waves.h"
//#include "AstroChart.h"
#include "astroCommon.h"
//#include "TopocentricCycle.h"
//#include "SwissEphemeris.subproj/Ephemeris.h"
//#include "Astro.bproj/AstroController.h"

static Waves	*sharedInstance = nil;

/* Private methods
 */
@interface Waves(PrivateMethods)
@end

@implementation Waves

/* create new instance of GeneralController
 * created: 2005-07-27
 */
+ (id)sharedInstance
{   //static Waves	*sharedInstance = nil;

    if (!sharedInstance)
        sharedInstance = [[Waves alloc] init];
    return sharedInstance;
}

- (id)init
{
    [super init];
    colors = [[NSArray arrayWithObjects:
        [NSColor colorWithCalibratedRed:0.0 green:0.0 blue:1.0 alpha:1.0], // blue
        [NSColor colorWithCalibratedRed:0.0 green:1.0 blue:1.0 alpha:1.0], // cyan
        [NSColor colorWithCalibratedRed:1.0 green:0.0 blue:0.0 alpha:1.0], // red
        [NSColor colorWithCalibratedRed:1.0 green:0.0 blue:1.0 alpha:1.0], // magenta
        nil] retain];
    return self;
}



/* return compression for x = [0, 1.0]
 * where 0 = nodal point, 1 = wave crest (50%)
 * created:  2005-06-28
 */
static double compression(double x)
{
    return 1.0 - sin(x * Pi);
}

/* remainder = num % denom
 */
double modulo(double num, double denom)
{
    return (num - floor(num / denom) * denom);
}

/* greatest common divisor
 */
int gcd(int a, int b)
{  int	t;

    while (b != 0)
    {   t = b;
        b = a % b;
        a = t;
    }
    return a;
}


/* return array of start degree + wave length [degree] for each wave found
 * The array is sorted by domination (number of nodal hits)
 * waves[0] = (270.0, 8.0, 11);
 * waves[1] = (180.0, 8.0,  4);
 *
 * created:  2005-07-27
 * modified: 2005-08-09
 */
- (NSArray*)wavesForDegreesInArray:(NSArray*)degArray
{   NSMutableArray	*waves = [NSMutableArray array];
    int			d, i, j, k, cnt;
    float		orb = 1.0 / 8.0;	// % of wave length
    float		ang, gap, pull, power, p, powCW, powCCW;
    int			skipCnt, skipIx[[degArray count]];

    /* Get power of harmonics between everything
     */
    for (d=2; d<180; d++)				// divider  (  2,   3,  4, ..., 180)
    {   float	waveLen = 360.0 / d;			// wave len (180, 120, 90, ...,   2)

        /* FIXME: only one set of harmonic waves and phase can survive on the ecliptic,
         * so we have to determine the strongest set (360-180-90-30 or 360-120-60-30-15-5-3)
         *   30 is supporting 60, 90, 120, 180, 360 as well as 15, 10, 7.5, 6, ...
         *   45 is supporting 90, 180, 360
         *   60 is supporting 120, 180, 360
         * or vice versa:
         *   12 is supporting 6, 4, 3, 2, 1 as well as 24, 36, 48, 60, ...
         *    8 is supporting 4, 2, 1       as well as 16, 24, 32, ...
         *    6 is supporting 3, 2, 1       as well as 12, 18, 24, ...
         *
         * possibilities:
         * - count the supportive dividers (this will leave the actual strength of these)
         * - for each divider, add the strength of all supportive (whole number) dividers
         *
         * - process (calc power, sort, sum) a complete harmonic set for each start wave ?
         *   1. loop for start (center) node of wave
         *   2. loop for dividers in groups of harmonic sets (2, 4, 6, 8, 10, ... or 3, 6, 9, 12, ...)
         *   3. loop for test points
         *      add up powers of set for each start point
         *      store entries as sets
         * - 2nd pass to weight each start point from the list with its harmonics
         */
        skipCnt = 0;
        for (i=0; i<[degArray count]-1; i++)		// start of wave
        {   float	degStart = [degArray floatAtIndex:i], deg;

            power = powCW = powCCW = 0.0;
            cnt = 0;
            for (j=0; j<[degArray count]; j++)		// test points to count
            {
                if (i == j)
                    continue;
                for (k=0; k<skipCnt; k++)
                    if ( j == skipIx[k] )	// index already used with this divider
                        break;
                if (k < skipCnt)
                    continue;

                /* The overlap (gap) of the wave length inside angle
                 * is equal to the rest of the division.
                 */
                deg = [degArray floatAtIndex:j];
                ang = angle(deg, degStart);		// angle = deg - degStart
                gap = modulo(ang, waveLen);		// overlap = angle % waveLen
                pull = ((gap > waveLen / 2.0) ? (gap-waveLen) : gap);	// pull to closer node, ahead = ccw = +
                gap = Min(gap, waveLen - gap);		// smallest overlap to this or next node
                if ( gap / waveLen < orb )
                {
                    power += (p = compression(gap / (waveLen/2.0)));	// add power of this node
                    if (pull >= 0.0) powCCW += p;	// pull ccw
                    if (pull <= 0.0) powCW  += p;	// pull cw
                    cnt ++;
                    skipIx[skipCnt++] = j;
                }
            }
            // FIXME: phase of wave should be averaged for equal pull to both sides to get correct power
            //powAv = 
            if (cnt > 2)
            {   int	ix;

                /* sort list of harmonics for power */
                for (ix=0; ix<[waves count]; ix++)	// sort for decreasing cnt
                {
                    if (power > [[waves objectAtIndex:ix] floatAtIndex:3])	// power * cnt
                    //if (cnt > [[waves objectAtIndex:ix] intAtIndex:4])	// cnt
                        break;
                }
                [waves insertObject:[NSArray arrayWithObjects:[NSNumber numberWithFloat:degStart],
                                                              [NSNumber numberWithFloat:waveLen],
                                                              [NSNumber numberWithInt:d],
                                                              [NSNumber numberWithFloat:power],
                                                              [NSNumber numberWithInt:cnt], nil]
                            atIndex:ix];
            }
        }
    }

    return waves;
}

/* return group with the calculated waves
 *
 * Components:
 * - Planets
 * - Ecliptic points
 * - House cups
 * - (Nodes)
 * - (Apogee)
 *
 * created:  2005-07-27
 * modified: 2005-07-28
 */
- (VGroup*)wavesForObjects:(double*)objs   cnt:(int)objCnt
                     nodes:(double*)nodes  cnt:(int)nodeCnt
                    houses:(double*)houses cnt:(int)houseCnt
                    center:(NSPoint)center radius:(float)radius amplitude:(float)amp
{   VGroup		*group = [VGroup group];
    VLine		*line;
    NSMutableArray	*degArray = [NSMutableArray array];
    NSArray		*waves;
    int			i;

    /* add ecliptic */
    [degArray addObject:[NSNumber numberWithFloat:  0.0]];
    [degArray addObject:[NSNumber numberWithFloat:180.0]];
    /* add planets */
    for ( i=0; i<objCnt; i++ )
        [degArray addObject:[NSNumber numberWithFloat:(float)objs[i]]];
    /* add nodes */
    for ( i=0; i<nodeCnt; i++ )
        [degArray addObject:[NSNumber numberWithFloat:(float)nodes[i]]];
    /* add house cups */
    for ( i=0; i<houseCnt; i++ )
        [degArray addObject:[NSNumber numberWithFloat:(float)houses[i]]];

    waves = [self wavesForDegreesInArray:degArray];

    /* draw waves or wave nodes */
    for (i=0; i<Min([waves count], 4); i++)	// draw max 4 waves !!!
    {   NSPoint	p0, p1;
        NSArray	*entry = [waves objectAtIndex:i];
        float	degStart, waveLen, deg0, deg;

        degStart = [entry floatAtIndex:0];
        waveLen  = [entry floatAtIndex:1];

        for (deg0 = 0; deg0 < 360.0; deg0 += waveLen)
        {
            deg = deg0 + degStart;

            /* draw triangles with tip at the node
             * for each divider we step inward, so that the many triangles come to the outside
             */

            /* draw line at node */
            p0 = vhfPointRotatedAroundCenter(NSMakePoint(center.x + radius,     center.y), deg, center);
            p1 = vhfPointRotatedAroundCenter(NSMakePoint(center.x + radius + 5, center.y), deg, center);
            line = [VLine lineWithPoints:p0 :p1];
            [line setColor:[colors objectAtIndex:Min(i, [colors count]-1)]];
            [group addObject:line];
        }
    }

    return group;
}



- (void)dealloc
{
    if (self == sharedInstance)
        return;
    [colors release];
    [super dealloc];
}

@end
