/* $Id: cbdmap.h,v 3.6 1991/04/22 22:05:02 putz Exp $ */
/*****************************************************************************
 * cdbmap.h - Header file for compressed binary map database (*.cbd) files
 *	These files encode features from a World Data Bank II map database.
 *
 * Originally created by Brian Reid (reid@pa.dec.com)
 * Modified March 1991 by Steve Putz (putz@parc.xerox.com)
 *****************************************************************************
The CIA World Bank II master information is a series of COBOL records, 
specifying 5,719,617 individual vectors, which
occupies about 130 megabytes of disk space.  The "cbd" files used by the
netmap program are compressed binary encodings of it, that 
collectively occupy about
15 megabytes of disk space.  These "cbd" files are produced by ciamap
and used by netmap

This database is available from the U.S. Government and is in the public domain.
The map database is divided into continents (North America, South America,
Europe, Africa, and Asia), and within continents it is divided into 
"Coastlines, Islands, and Lakes" (cil files), "Rivers" (riv files),
"International political boundaries" (bdy files) and "National political
boundaries" (pby files). Each file is divided into several thousand
"segments", and each segment is divided into some number of "strokes".

The .cbd encoding of the database encodes each continent into a
directory and each division into a file.  In each file there is a 
header record, a series of segments and strokes, and a segment index.
*****************************************************************************/

#ifndef	__cdbmap_h
#define	__cdbmap_h

#define	BIT32	int		/* change these definitions to suit your */
#define	BIT16	short		/* own machine				 */
#define	BIT8	char

#define SEC_PER_DEG	(60*60)
#define DEG90_SECS	(90*SEC_PER_DEG)
#define DEG180_SECS	(180*SEC_PER_DEG)

struct cbdhead {
	long	magic;		/* Magic number */
	long	dictaddr;	/* Offset of segment dictionary in file */
	long	segcount;	/* Number of segments in file */
	long	segsize;	/* Size of segment dictionary */
	long	segmax;		/* Size of largest segment's strokes, /2 */
    /* the following apply to CBD_MAGIC2 only */
	long	maxlat,minlat,maxlong,minlong;	/* Bounding box of map */
	long	features;	/* bits indicate feature "ranks" present */
	long	scale_shift;	/* bits to shift coordinate data */
	long	lat_offset;	/* lattitude offset */
	long	lng_offset;	/* longitude offset */
};

#define	CBD_MAGIC	0x20770002
#define	CBD_HEADSIZE1	40		/* size of old header */
#define	CBD_MAGIC2	0x20770033
#define	CBD_HEADSIZE2	(sizeof(struct cbdhead))

/* Segment header.  One per segment, followed by data compressed strokes.
 */
struct seghead {
	BIT32   orgx,orgy;	/* Origin of first stroke in segment */
	BIT32   id;		/* Segment identifier */
	BIT16   nstrokes;	/* How many strokes in the segment follow */
};

/* The data-compression scheme uses integer seconds to represent lat/long,
 * and stores each stroke as a [dx,dy] from the previous point. If dy will
 * fit into 8 bits and dx into 7 bits, then the entire [dx,dy] is stored in a
 * 16-bit field with bit 0x4000 turned on as a flag. If either value is too
 * large for that scheme, then both are stored as 32-bit values, with the
 * 0x40000000 bit turned off (even in negative numbers) in the first of them.
 */
#define	MAX8y	((short) 0x7F) /* largest signed 8-bit value */
#define	MAX8x	((short) 0x3F) /* largest signed 8-bit value */
#define	SHORTFLAG 0x4000       /* flag saying this is a short stroke */
#define	SHORTBYTE 0x40         /* flag saying this is a short stroke */

/*
 * Segment dictionary sits on the end of the file, is pointed to by the
 * file header, and points to the segment headers.
 */
struct segdict {
	BIT32   segid;		/* Mysterious 7-digit number */
	BIT32   maxlat,minlat,maxlong,minlong;	/* Bounding box of strokes */
	BIT32   absaddr;	/* Address in file of segment header */
	BIT16   nbytes;		/* # bytes of strokes that follow */
	BIT16   rank;		/* Type of feature this segment draws */
};

#endif	/* !__cdbmap_h */

/*****************************************************************************
The "rank" is an indication of what the segment depicts:

In "Boundary" files (bdy):
	01	Demarcated or delimited boundary
	02	Indefinite or in Dispute
	03	Other line of separation of soverignity on land

In "Coast, Islands and Lakes" files (cil)
	01	Coast, islands and lakes that appear on all maps
	02	Additional major islands and lakes
	03	Intermediate islands and lakes
	04	Minor islands and lakes
	06	Intermittent major lakes
	07	Intermittent minor lakes
	08	Reefs
	09	Salt pans -- major
	10	Salt pans -- minor
	13	Ice Shelves -- major
	14	Ice Shelves -- minor
	15	Glaciers

In "Rivers" files (riv)
	01	Permanent major rivers
	02	Additional major rivers
	03	Additional rivers
	04	Minor rivers
	05	Double lined rivers
	06	Intermittent rivers -- major
	07	Intermittent rivers -- additional
	08	Intermittent rivers -- minor
	10	Major canals
	11	Canals of lesser importance
	12	Canals -- irrigation type
*****************************************************************************/
