/* colorFunctions.m
 * color converting, filtering, etc.
 *
 * Copyright (C) 2003-2004 by vhf interservice GmbH
 * Author:   Georg Fleischmann
 *
 * created:  2003-08-03
 * modified: 2004-06-02
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 *
 * vhf interservice GmbH, Im Marxle 3, 72119 Altingen, Germany
 * eMail: info@vhf.de
 * http://www.vhf.de
 */

//#include <math.h>
#include <VHFShared/types.h>
#include "colorFunctions.h"

RGB RGBFromHSI(HSI hsi)
{   RGB	rgb;

    if (hsi.h < 0.0)      hsi.h = 0.0;
    else if (hsi.h > 1.0) hsi.h = 1.0;
    if (hsi.s < 0.0)      hsi.s = 0.0;
    else if (hsi.s > 1.0) hsi.s = 1.0;
    if (hsi.i < 0.0)      hsi.i = 0.0;
    else if (hsi.i > 1.0) hsi.i = 1.0;

    {   int	I = (int)(hsi.h * 6);
        float	V = hsi.i;
        float	S = hsi.s;
        float	F = (hsi.h * 6) - I;
        float	M = V * (1 - S);
        float	N = V * (1 - S * F);
        float	K = M - N + V;

        switch (I)
        {
            default: rgb.r = V; rgb.g = K; rgb.b = M; break;
            case 1:  rgb.r = N; rgb.g = V; rgb.b = M; break;
            case 2:  rgb.r = M; rgb.g = V; rgb.b = K; break;
            case 3:  rgb.r = M; rgb.g = N; rgb.b = V; break;
            case 4:  rgb.r = K; rgb.g = M; rgb.b = V; break;
            case 5:  rgb.r = V; rgb.g = M; rgb.b = N;
        }
    }
    return rgb;
}
HSI HSIFromRGB(RGB rgb)
{   HSI		hsi;
    float	min, max, delta;

    if (rgb.r < 0.0)      rgb.r = 0.0;
    else if (rgb.r > 1.0) rgb.r = 1.0;
    if (rgb.g < 0.0)      rgb.g = 0.0;
    else if (rgb.g > 1.0) rgb.g = 1.0;
    if (rgb.b < 0.0)      rgb.b = 0.0;
    else if (rgb.b > 1.0) rgb.b = 1.0;

    min = Min( Min(rgb.r, rgb.g), rgb.b );
    max = Max( Max(rgb.r, rgb.g), rgb.b );
    hsi.i = max;					// hue
    delta = max - min;

    hsi.s = ( max != 0 ) ? delta / max : 0.0;		// saturation
    if (hsi.s == 0.0)					// hue is undefined
        hsi.h = 0.0;

    if ( rgb.r == max )
        hsi.h = ( rgb.g - rgb.b ) / delta;		// between yellow & magenta
    else if ( rgb.g == max )
        hsi.h = 2.0 + ( rgb.b - rgb.r ) / delta;	// between cyan & yellow
    else
        hsi.h = 4.0 + ( rgb.r - rgb.g ) / delta;	// between magenta & cyan

    hsi.h /= 6.0;					// normalize hue [0, 1]
    if ( hsi.h < 0.0 )  hsi.h += 1.0;
    if ( hsi.h >= 1.0 ) hsi.h -= 1.0;

    return hsi;
}


/* filter variable number of HSI colors
 */
HSI filterColors(int count, ...)
{   va_list	args;
    int		i;
    RGB		rgb1, rgb;

    va_start(args, count);
    rgb = RGBFromHSI(va_arg(args, HSI));
    for ( i=1; i<count; i++ )
    {
        rgb1 = RGBFromHSI(va_arg(args, HSI));
        rgb.r *= rgb1.r;
        rgb.g *= rgb1.g;
        rgb.b *= rgb1.b;
    }
    va_end(args);
    return HSIFromRGB(rgb);
}

/* add variable number of HSI colors
 * eg: r + g + b = white
 */
HSI addColors(int count, ...)
{   va_list	args;
    int		i;
    RGB		rgb1, rgb;

    va_start(args, count);
    rgb = RGBFromHSI(va_arg(args, HSI));
    for ( i=1; i<count; i++ )
    {
        rgb1 = RGBFromHSI(va_arg(args, HSI));
        rgb.r += rgb1.r;
        rgb.g += rgb1.g;
        rgb.b += rgb1.b;
    }
    va_end(args);
    return HSIFromRGB(rgb);
}
