/* astroCommon.m
 * Common astronomical functions
 *
 * Copyrigth (C) 1998-2003 by vhf interservice GmbH
 * Author:   Georg Fleischmann
 *
 * created:  1998-11-06
 * modified: 2003-07-05
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 *
 * vhf interservice GmbH, Im Marxle 3, 72119 Altingen, Germany
 * eMail: info@vhf.de
 * http://www.vhf.de
 */

#include <Foundation/Foundation.h>
#include <math.h>
#include <VHFShared/types.h>
#include <VHFShared/VHFStringAdditions.h>
#include "astroCommon.h"

/* ascending Sun's Node
 * Intersection between the Sun's equator and the ecliptic.
 * the descending node is always 180 degree to the ascending node.
 */
float sunNode(int year)
{
    return 73.66670 + (float)(year - 1850) * 0.01396;
}

NSMutableData* reducedTable(NSData *tableData, int tableSize, double factor)
{   int			i;
    NSMutableData	*nTableData = [NSMutableData dataWithLength:tableSize * sizeof(double)];
    double		*table = (double*)[tableData bytes], *nTable = [nTableData mutableBytes];

    /* reduce  */
    for ( i=0; i<tableSize; i++ )
        nTable[i] = table[i] * factor;
    return nTableData;
}

/* [0, 1] or [-1, 1]
 */
NSMutableData* normalizedTable(NSData *tableData, int tableSize, BOOL positive)
{   int			i;
    double		min, max, absMax;
    NSMutableData	*nTableData = [NSMutableData dataWithLength:tableSize * sizeof(double)];
    double		*table = (double*)[tableData bytes], *nTable = [nTableData mutableBytes];

    /* get max, sum */
    for ( i=0, min=LARGE_COORD, max=LARGENEG_COORD; i<tableSize; i++ )
    {
        min = Min(table[i], min);
        max = Max(table[i], max);
    }

    absMax = Max(Abs(max), Abs(min));
    if (absMax == 0.0)
        absMax = 1.0;

    /* normalize  */
    for ( i=0; i<tableSize; i++ )
    {
        if (positive)
        {
            nTable[i] = table[i] - min;			// move to 0
            if (max-min)
                nTable[i] = nTable[i] * 1.0 / (max-min);	// [0, 1]
        }
        else
            nTable[i] = table[i] / absMax;		// [-1, 1]
    }
    return nTableData;
}

/*
 * 1 2 3 4 ...
 *   2   4 ...
 *     3   ...
 *       4 ...
 *
 * range = [0, 1]
 */
double *harmonicTable(int tableSize)
{   double	*table = NSZoneMalloc(NULL, tableSize * sizeof(double));
    int		i, j, a;

    for ( i=0; i<tableSize; i++ )
        table[i] = 0.0;

    for ( i=1; i<=tableSize; i++ )
    {   int	cnt = 0;

        for ( j=1; j<=i; j++ )
            if ( !(i % j) )
                cnt++;
        //table[i-1] = (double)cnt / (double)i;		// numbers
        /* set angles */
        for ( a=tableSize/i; a<=tableSize; a+=tableSize/i )
        {   int	a1 = (a>=tableSize) ? 0 : a;
            if ( !table[a1] )
                table[a1] = (double)cnt / (double)i;
        }
    }

    return table;
}

/* linear function (analog sinus)
 *   0  0
 *  90  1
 * 180  0
 * 270 -1
 */
double lin(double v)
{
    while (v >= 360.0)
        v -= 360.0;
    if (v <= 90.0)
        return 0 + v / 90.0;
    if (v <= 180.0)
        return 1 - (v-90) / 90.0;
    if (v <= 270.0)
        return 0 - (v-180) / 90.0;
    return -1 + (v-270) / 90.0;
}
/* calculate harmonic pattern
 * this is the sum of the harmonics of a longitudinal wave
 * range = [-1, 1]
 */
double *harmonicPattern(int tableSize, int depth)
{   double	*table = NSZoneMalloc(NULL, tableSize * sizeof(double));
    double	max;
    int		d, i, t;

    //for ( d=0.0; d<tableSize; d+=1.0)
    //    table[(int)d] = harmonicValue(d*360.0/(double)tableSize);

//#if 0
    for ( i=0; i<tableSize; i++ )
        table[i] = 0.0;

    /* basic curve */
    //for ( t=0; t<tableSize; t+=1 )
    //    table[t] += Sin(t);

    /* harmonics: integer deviders 1, 2, 3, 4, 5, ...
     * fundamental and every harmonic is devided recursively
     */
    /*for ( d=1; d<=depth; d*=2 )		// harmonics
    {
        for ( t=0; t<tableSize; t+=1 )		// table column
            table[t] += Abs(Sin(t * d * 360.0/tableSize)) / d;
    }*/

    /* harmonics: integer deviders 1, 2, 3, 4, 5, ... */
    for ( d=1; d<=depth; d+=1 )			// harmonic
    {
        for ( t=0; t<tableSize; t+=1 )		// table column
            table[t] += Abs(Sin(t * (double)d * 360.0/tableSize)) / d;
    }

    /* get max, sum */
    for ( i=0, max=0.0; i<tableSize; i++ )
        max = Max(Abs(table[i]), max);

    /* normalize, invert  */
    for ( i=0; i<tableSize; i++ )
        table[i] = 1.0 - (table[i] / max);	// [0, 1]
//#endif

    /* invert 2nd part
     */
    /*for ( i=0; i<tableSize; i++ )
    {
        if (i<=tableSize/2)
            table[i] *= 0.5;
        else
            table[i] = (1.0 - table[i] + 1.0) * 0.5;
    }*/

    /* add basic curve, and normalize again */
    /*for ( t=0; t<tableSize; t+=1 )
        table[t] = Sin(t) + table[t]/2.0;
    for ( i=0, max=0.0; i<tableSize; i++ )
        max = Max(Abs(table[i]), max);
    for ( i=0; i<tableSize; i++ )
        table[i] = table[i] / max;*/		// [-1, 1]

    return table;
}

/* deg = [0, 360]
 */
double harmonicValue(double deg)
{   double	d, max = 5.88104575996997259 /*4.414657969489995*/, v = 0.0;

    /* harmonics: integer deviders 1, 2, 3, 4, 5, ... */
    for ( d=1.0; d<=3600.0; d+=1.0 )	// harmonic
        v += Abs(Sin(deg * d)) / d;
    v = 1.0 - (v/max);			// normalize, invert
    return v;
}

/* calculate harmonic pattern
 * 360 - 180 -> 120, 240
 * 180 -  90 ->  60, 300
 *  90 -  45 ->  30, 330, 150, 210
 * range = [-1, 1]
 */
double *harmonicPattern1(int tableSize)
{   double	*table = NSZoneMalloc(NULL, tableSize * sizeof(double));
    double	o, lastO = 0.0;
    int		i;

    for ( i=0; i<tableSize; i++ )
        table[i] = 0.0;

    /* octaves */
    for ( o=360.0; o>=360.0/(double)tableSize; o/=2.0 )
    {
        table[(o>=360.0) ? 0 : (int)(o*(double)tableSize/360.0)] = 1.0 * o/360.0;	// octave
        if ( o<360.0 )
        {   double	h = (lastO - o) / 3.0, seg;

            for ( seg = o; seg<360.0; seg += lastO )
            {
                if ( table[(int)(seg*(double)tableSize/360.0)] == 0.0 )
                    table[(int)(seg*(double)tableSize/360.0)] = 1.0 * o/360.0;	// octave
                if ( table[(int)(seg+h)] == 0.0 )
                    table[(int)(seg+h)] = -1.0 * o/360.0;
                if ( table[(int)(seg-h)] == 0.0 )
                    table[(int)(seg-h)] = -1.0 * o/360.0;
            }
        }
        lastO = o;
    }

    return table;
}

/* revert angles more than 180 degrees from ref
 * a    ref
 * -10  0   -> -10
 * -185 0   -> 175
 */
/*float comparableAngle180( float a, float ref)
{
    if ( a < ref && a < ref-180.0 )
        a += 360.0;
    else if ( a > ref && a > ref+180.0 )
        a -= 360.0;
    return a;
}*/

/* equal angles
 * Angle from sun to moon SunMoon: deg1=Sun, deg2=Moon
 */
/*float angleFromFirstToSecond( float deg1, float deg2 )
{   float	a;

    if ( deg1 < deg2 )
        deg1 += 360.0;
    a = deg2 - deg1;
    if ( a > 180.0 )
        a -= 360.0;
    else if ( a < -180.0 )
        a += 360.0;
    return a;
}*/

/*float angleBetween( float deg1, float deg2 )
{   float	angle;

    angle = Diff(deg1, deg2);
    if ( angle > 180.0)
        angle = 360.0 - angle;
    return angle;
}*/

/* 137.1234 -> 137 7'24"
 */
NSString *stringFromDeg(float deg)
{   int		d, m, s;
    NSString	*string;

    d = (int)deg;
    m = (int)floor((deg-d) * 60.0);
    s = (int)floor((deg-d-m/60.0) * 3600.0 + 0.5);
    string = [NSString stringWithFormat:@"%@%@'%@\"",
                       [[NSString stringWithFormat:@"%d", d] stringWithLength:-4],
                       [[NSString stringWithFormat:@"%d", m] stringWithLength:-2],
                       [[NSString stringWithFormat:@"%d", s] stringWithLength:-2] ];
//NSLog(@"degFromFloat: %f -> %@", deg, string);
    return string;
}

/* "430'00" -> 4.5 or 4.5
 */
float degFromString(NSString *degString)
{   float		deg = 0.0;
    NSScanner		*scanner = [NSScanner scannerWithString:degString];
    NSCharacterSet	*digitSet = [NSCharacterSet decimalDigitCharacterSet];
    NSCharacterSet	*degreeSet = [NSCharacterSet characterSetWithCharactersInString:@" '\""];
    NSString		*string;
    NSRange		range;

    range = [degString rangeOfCharacterFromSet:degreeSet];
    if ( !range.length )
        deg = [degString floatValue];
    else
    {
        if ( [scanner scanCharactersFromSet:digitSet intoString:&string] )
            deg += [string floatValue];
        [scanner scanUpToCharactersFromSet:digitSet intoString:NULL];
        if ( [scanner scanCharactersFromSet:digitSet intoString:&string] )
            deg += [string floatValue] / 60.0;
        [scanner scanUpToCharactersFromSet:digitSet intoString:NULL];
        if ( [scanner scanCharactersFromSet:digitSet intoString:&string] )
            deg += [string floatValue] / 3600.0;
    }
    return deg;
}

/* created: 21.08.97
 */
int sortAsNumbers(id str1, id str2, void *context)
{
    if ( [str1 intValue] < [str2 intValue] )
        return NSOrderedAscending;
    if ( [str1 intValue] > [str2 intValue] )
        return NSOrderedDescending;
    return NSOrderedSame;
}

/* MJD: modified julian date
 */
double mjd( NSCalendarDate *date )
{   int		year, month, day;
    double	hour, mjd;
    double	a, b;

    year = [date yearOfCommonEra];
    month = [date monthOfYear];
    day = [date dayOfMonth];
    hour = (double)[date hourOfDay]+[date minuteOfHour]/60.0+[date secondOfMinute]/3600.0;

    a = 10000.0*year+100.0*month+day;
    if ( month<=2 )
    {
        month += 12;
        year--;
    }
    if ( a<=15821004.1 )
        b = -2.0+floor((year+4716.0)/4.0)-1179.0;
    else
        b = floor(year/400.0)-floor(year/100.0)+floor(year/4.0);
    a = 365.0*year-679004.0;
    mjd = a+b+floor(30.6001*(month+1.0))+day+hour/24.0;
    return mjd;
}

NSCalendarDate *dateFromMJD(double mjd)
{   int			year, month, day, hour, min, sec;
    int			b, d, f;
    double		jd, jd0, c, e;
    NSCalendarDate	*date;

    jd = mjd+2400000.5;
    jd0 = floor(jd+0.5);
    if ( jd0<2299161.0 )
        c = jd0+1524.0;
    else
    {
        b = floor((jd0-1867216.25)/36524.25);
        c = jd0+(b-floor(b/4.0))+1525.0;
    }
    d = floor((c-122.1)/365.25);
    e = 365.0*d+floor(d/4.0);
    f = floor((c-e)/30.6001);
    day = floor(c-e+0.5)-floor(30.6001*f);
    month = f-1-12*floor(f/14.0);
    year = d-4715-floor((7.0+month)/10.0);
    hour = 24.0*(jd+0.5-jd0);
    min = (24.0*(jd+0.5-jd0) - hour)/60.0;
    sec = ((24.0*(jd+0.5-jd0) - hour) - min*60.0)/3600.0;

    date = [NSCalendarDate dateWithYear:year month:month day:day hour:hour minute:min second:sec timeZone:[NSTimeZone timeZoneWithName:@"GMT"]];
    return date;
}

/* gmt to local time
 * start at 22.09. 00:00 UTC
 * each 24h (86400s) +4 minutes (star day = 23:56:04.1) -> +235.9 sec
 * //we also add a correction of 9.8s per hour of day
 */
float frac(float x)
{
    x -= floor(x);
    if (x<0) x += 1.0;
    return x;
}
NSCalendarDate *sideralTime(NSCalendarDate *gmt, float longitude)
{   double	mj, mjd0, t, ut, gmst, lmst;

    longitude = -longitude;

    mj = mjd(gmt);
//NSLog(@"gmt:%@ gmt1:%@", gmt, dateFromMJD(mj));

    mjd0 = floor( mj );
    ut = (mj-mjd0)*24;
    t = (mjd0-51544.5)/36525.0;
    gmst = 6.697374558+1.0027379093*ut+(8640184.812866+(0.093104-6.2e-6*t)*t)*t/3600.0;
    lmst = 24.0*frac((gmst-longitude/15.0)/24.0);

    gmt = [NSCalendarDate dateWithString: [gmt descriptionWithCalendarFormat:@"%Y%m%d 00:00:00 %z"]
                          calendarFormat:@"%Y%m%d %H:%M:%S %z"];
    return [gmt dateByAddingYears:0 months:0 days:0 hours:0 minutes:0 seconds:(int)(lmst*3600.0)];
}
float longitudeWithSideralTime(NSCalendarDate *st, NSCalendarDate *gmt)
{   double	mj, mjd0, t, ut, gmst, lmst, lon;

    mj = mjd(gmt);
    mjd0 = floor( mj );
    ut = (mj-mjd0)*24;
    t = (mjd0-51544.5)/36525.0;
    gmst = 6.697374558+1.0027379093*ut+(8640184.812866+(0.093104-6.2e-6*t)*t)*t/3600.0;

    gmt = [NSCalendarDate dateWithString: [gmt descriptionWithCalendarFormat:@"%Y%m%d 00:00:00 %z"]
                          calendarFormat:@"%Y%m%d %H:%M:%S %z"];
    lmst = [st timeIntervalSinceDate:gmt] / 3600.0;
    lon = (lmst - gmst) * 15.0;
    while (lon <= -180.0)
        lon += 360.0;
    while (lon > 180.0)
        lon -= 360.0;

    return lon;
}

float ra( NSCalendarDate *sideralTime)
{   float	s, ra;

    s = [sideralTime timeIntervalSinceDate:[NSCalendarDate dateWithString:[sideralTime descriptionWithCalendarFormat:@"%Y.%m.%d-00:00 %z"] calendarFormat:@"%Y.%m.%d-%H:%M %z"]];
    ra = s / (4.0*60.0);
    return ra;
}

float mc( NSCalendarDate *sideralTime)
{   float	s, e = 23.4333, l, mc, ra;

    s = [sideralTime timeIntervalSinceDate:[NSCalendarDate dateWithString:[sideralTime descriptionWithCalendarFormat:@"%Y.%m.%d-00:00 %z"] calendarFormat:@"%Y.%m.%d-%H:%M %z"]];
    ra = s / (4.0*60.0);
    l = Cos(e)*Cot(ra);
    mc = Acot(l);
    if ( mc<0 )
        mc += 360.0;

    /* from 12:00 - 00:00 sideralTime -> mc = 180 - 360 degree
     * from 00:00 - 12:00 sideralTime -> mc = 0 - 180 degree
     * -06:00 -> mc
     * 06:00- -> mc -= 180.0
     */
    if ( s>=0 && s<43200 )
    {
        if ( mc > 180.0 )
            mc -= 180.0;
    }
    else if ( mc < 180.0 )
        mc += 180.0;

//NSLog(@"in -> st=%@ s=%f, ra=%f, l=%f, mc=%f", [sideralTime descriptionWithCalendarFormat:@"%Y%m%d%H%M%S %Z"], s, ra, l, mc);
    return mc;
}
NSCalendarDate *sideralTimeWithMC(float mcDeg, NSCalendarDate *utc)
{   float		s, e = 23.4333, l, mc, ra;
    NSCalendarDate	*st;

    mc = mcDeg;
    /*if (mcDeg > 180.0)
        mc -= 180.0;*/

    l = Cot(mc);
    ra = Acot(l / Cos(e));
    if (mcDeg >= 90.0 && mcDeg < 270.0)
        ra += 180.0;
    else if (mcDeg >= 270)
        ra += 360.0;
    s = ra * (4.0*60.0);
    st = [[NSCalendarDate dateWithString:[utc descriptionWithCalendarFormat:@"%Y%m%d 00:00:00 %z"]
                          calendarFormat:@"%Y%m%d %H:%M:%S %z"]
          dateByAddingYears:0 months:0 days:0 hours:0 minutes:0 seconds:s];
//NSLog(@"ou -> st=%@ s=%f, ra=%f, l=%f, mc=%f", [st descriptionWithCalendarFormat:@"%Y%m%d%H%M%S %Z"], s, ra, l, mc);

    return st;
}

float ac( NSCalendarDate *sideralTime, float lat)
{   float	s, l, e = 23.4333, ac, oa;

    s = [sideralTime timeIntervalSinceDate:[NSCalendarDate dateWithString:[sideralTime descriptionWithCalendarFormat:@"%Y.%m.%d-00:00-%z"] calendarFormat:@"%Y.%m.%d-%H:%M-%z"]];
    oa = s / (4.0*60.0) + 90.0;
    l = Cos(e)*Cot(oa) - Sin(e)*Tan(lat)/Sin(oa);
    ac = Acot(l);
    if ( ac<0 )
        ac += 360.0;
    /* from 18:00 - 06:00 sideralTime -> ac = 0 - 180 degree
     * from 06:00 - 18:00 sideralTime -> ac = 180 - 360 degree
     */
    if ( s>=21600 && s<64800 )
    {
        if ( ac<180.0 )
            ac += 180.0;
    }
    else if ( ac>180.0 )
        ac -= 180.0;

//NSLog(@"in -> ac=%f, l=%f, st=%@ lat=%f", ac, l, sideralTime, lat);
    return ac;
}

/* the opposite direction for AC/MC
 */
float longitudeForMC(float mcDeg, NSCalendarDate *utc)
{   NSCalendarDate	*sideralTime;
    float		lon;

    sideralTime  = sideralTimeWithMC(mcDeg, utc);
    lon = longitudeWithSideralTime(sideralTime, utc);
    return lon;
}
float latitudeForAC(float acDeg, NSCalendarDate *sideralTime)
{   double	s, l, e = 23.4333, oa, lat;

    s = [sideralTime timeIntervalSinceDate:[NSCalendarDate dateWithString:[sideralTime descriptionWithCalendarFormat:@"%Y.%m.%d-00:00-%z"] calendarFormat:@"%Y.%m.%d-%H:%M-%z"]];
    oa = s / (4.0*60.0) + 90.0;
    /* FIXME: if acDeg == 0.0 -> l gets infinity */
    l = Cot(acDeg);
    lat = Atan((Cos(e)*Cot(oa) - l) * Sin(oa) / Sin(e));

//NSLog(@"ou -> ac=%f, l=%f, st=%@ lat=%f", acDeg, l, sideralTime, lat);
    return lat;
}

/* convert planet string to planet symbol
 */
NSString *planetSymbol(NSString *planet)
{
    if ( [planet isEqual:@"Sun"] )
        return @"a";
    if ( [planet isEqual:@"Mercury"] )
        return @"b";
    if ( [planet isEqual:@"Venus"] )
        return @"c";
    if ( [planet isEqual:@"Mars"] )
        return @"d";
    if ( [planet isEqual:@"Jupiter"] )
        return @"e";
    if ( [planet isEqual:@"Saturn"] )
        return @"f";
    if ( [planet isEqual:@"Uranus"] )
        return @"g";
    if ( [planet isEqual:@"Neptune"] )
        return @"h";
    if ( [planet isEqual:@"Pluto"] )
        return @"i";
    if ( [planet isEqual:@"Chiron"] )
        return @"j";
    if ( [planet isEqual:@"Lilith"] || [planet isEqual:@"Apogee"] || [planet isEqual:@"True Apogee"] )
        return @"l";
    if ( [planet isEqual:@"Moon"] )
        return @"k";
    if ( [planet isEqual:@"Node"] || [planet isEqual:@"True Node"] )
        return @"m";
    if ( [planet isEqual:@"DC"] )
        return @"N";
    if ( [planet isEqual:@"IC"] )
        return @"O";
    if ( [planet isEqual:@"AC"] )
        return @"P";
    if ( [planet isEqual:@"MC"] )
        return @"Q";
    if ( [planet isEqual:@"R"] )
        return @"\x60";
    return @"?";
}

NSString *nodeSymbol(NSString *planet)
{
    if ( [planet isEqual:@"Mercury"] )
        return @"R";
    if ( [planet isEqual:@"Venus"] )
        return @"S";
    if ( [planet isEqual:@"Mars"] )
        return @"T";
    if ( [planet isEqual:@"Jupiter"] )
        return @"U";
    if ( [planet isEqual:@"Saturn"] )
        return @"V";
    if ( [planet isEqual:@"Uranus"] )
        return @"W";
    if ( [planet isEqual:@"Neptune"] )
        return @"X";
    if ( [planet isEqual:@"Pluto"] )
        return @"Y";
    if ( [planet isEqual:@"Chiron"] )
        return @"Z";
    if ( [planet isEqual:@"Moon"] )
        return @"m";
    return @"?";
}

/* sign = 0 - 11
 */
NSString *zodiacSign(int sign)
{
    switch (sign)
    {
        case 0:		// Aries !
            return @"A";
        case 1:		// Taurus
            return @"B";
        case 2:		// Gemini
            return @"C";
        case 3:		// Cancer !
            return @"D";
        case 4:		// Leo
            return @"E";
        case 5:		// Virgo
            return @"F";
        case 6:		// Libra
            return @"G";
        case 7:		// Scorpio
            return @"H";
        case 8:		// Sagittarius
            return @"I";
        case 9:		// Capricorn
            return @"J";
        case 10:	// Aquarius
            return @"K";
        case 11:	// Pisces
            return @"L";
        default:
            return @"?";
    }
}

/* aspect symbol
 * deg = [0, 360]
 */
NSString *aspectSymbol(float deg, float orb)
{
    if (Diff(deg, 0.0) < orb || Diff(deg, 360.0) < orb)		// 0, 360   -> conjunct
        return @"p";
    if (Diff(deg, 30.0) < orb || Diff(deg, 330.0) < orb)	// 30, 330
        return @"q";
    if (Diff(deg, 45.0) < orb || Diff(deg, 315.0) < orb)	// 45, 315  -> semi square
        return @"r";
    if (Diff(deg, 60.0) < orb || Diff(deg, 300.0) < orb)	// 60, 300  -> sextile
        return @"s";
    if (Diff(deg, 72.0) < orb || Diff(deg, 288.0) < orb)	// 72, 288  -> septil
        return @"t";
    if (Diff(deg, 90.0) < orb || Diff(deg, 270.0) < orb)	// 90, 270  -> square
        return @"u";
    if (Diff(deg, 120.0) < orb || Diff(deg, 240.0) < orb)	// 120, 240 -> trigon
        return @"v";
    if (Diff(deg, 135.0) < orb || Diff(deg, 225.0) < orb)	// 135, 225 -> semi square
        return @"w";
    if (Diff(deg, 144.0) < orb || Diff(deg, 216.0) < orb)	// 144, 216 -> septil
        return @"x";
    if (Diff(deg, 150.0) < orb || Diff(deg, 210.0) < orb)	// 150, 210
        return @"y";
    if (Diff(deg, 180.0) < orb)					// 180 -> opposition
        return @"z";
    return nil;
}

/* angle = object1 - object2
 */
double angle( double deg1, double deg2 )
{   double angle = deg1 - deg2;

    while ( angle >= 360.0 )
        angle -= 360.0;
    while ( angle < 0.0 )
        angle += 360.0;
    return angle;
}
/* mirror = object1 - (360 - object2)
 */
double mirror( double deg1, double deg2 )
{
    return angle( deg1, 360.0 - deg2 );
}
/* harmony = 180 * (obj - obj1) / (obj2 - obj1)
 * obj1 represents 0 degree, obj2 represents 180 degree
 * we return the angle obj represents between obj1 and obj2
 */
double harmony( double obj, double obj1, double obj2)
{   double	a, b;

    a = angle(obj2, obj1);
    b = angle(obj, obj1);
    if (b < a && a >= 0.0001)
        return 180.0 * b / a;
    return -180.0 * (180.0 - b) / (360.0 - a);
}
