/* CIAMap.h
 * Astrological map for Cenon based on the CIA maps
 *
 * Copyright (C) 2001-2006 by vhf interservice GmbH
 * Author:   Ilonka Fleischmann
 *
 * created:  2001-07-30
 * modified: 2006-04-05 (projection added)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 */

#ifndef VHF_H_CIAMAP
#define VHF_H_CIAMAP

#include <AppKit/AppKit.h>
#include "cbdmap.h"
#include "../astroLocations.h"

/*
 * interface for vector map
 */

#define PROJ_RECT	0		/* equirectangular */
#define PROJ_ELLIPSE	1		/* elliptical */
#define PROJ_SINE	2		/* sinusoidal */
#define PROJ_MERCATOR	3		/* mercator circle */
#define PROJ_AZIMUT	4		/* azimutal stereographic */
#define MAX_PROJ	PROJ_AZIMUT   /* PROJ_SINE */

#define MAX_MAPS	1000
#define MAX_TYPES	50
#define MAX_SCALES	25
#define MAX_COLORS	256		/* absolute maximum is 256 */

#define UNDEF_DEGS	-1000

typedef struct MRect
{
    int	r_width;
    int	r_height;
    int	r_top;
    int	r_left;
    int	r_right;
} MRect;

typedef struct colormap_t
{
    int			type;
    long		length;
    unsigned char	map[5][MAX_COLORS];
} colormap_t;

typedef struct pr_pos
{
    float	x;
    float	y;
} pr_pos;

typedef struct MapFile
{
    int		used;
    char	*dir, *name;
    FILE	*file;
    struct	cbdhead header;	// file header
} MapFile;

typedef struct MapStruct
{
    int		scaleI, type;
    MapFile	*mfile;
    int		rank_offset;	// delta for feature numbers
    long	minSSec, maxNSec, minWSec, maxESec;	// in seconds
    float	minLat, maxLat, minLng, maxLng;		// in degrees
    int		uSecShift;	// left shift for units to seconds
    long	uPerDeg;	// map units per degree
    long	cntrLngU;	// current center longitude in units
    long	muPerDot;	// current scale to magified units
} MapStruct;

typedef struct SegList
{
    struct pr_pos	*ptlist;
    unsigned char	*mvlist;
    int			maxpoints;
    int			op;
    int			line_width;
    short		*pattern;
} SegList;

typedef struct MapFeature
{
    int 	rank;
    char	*name;
    int		line_width;
    int		color;		// colormap index
    int		highlight;	// highlight flag
} MapFeature;

typedef struct MapStateStruct
{
    /* To change the map area displayed, set the following before calling map_setup.
     * The values are adjusted by map_setup to fit the frame
     */
    float	centerLng, centerLat, showLng, showLat;

    /* To change the map appearance, set the following before calling map_draw */
    int		proj;		// PROJ_RECT, PROJ_ELLIPSE, PROJ_SINE
    int		useColor;	// use colors if non-zero
    int		margin;		// pixel margin around unclipped map
    int		gridLines;	// #of graticule lines, 0=outline, -1=none
    int		scaleI;		// map subset to use (set by map_setup)

    float	gridDegrees;	// set after graticule displayed

    /* The following are set by map_init */
    float	def_centerLng, def_centerLat, def_showLng, def_showLat;
    int		debug;		// 0 = none, 1 = verbose, -1 = debug all
    int		mCount, mTypeCount, scaleCount;
    char	*mTypes[MAX_TYPES], *scaleNames[MAX_SCALES];	// indexed by [mapType]
    char	maxFeature[MAX_TYPES];			// [mapType]
    MapFeature	*features[MAX_TYPES];			// [mapType]
    long 	min_featureMasks[MAX_TYPES];		// [mapType]
    long	enableMasks[MAX_TYPES];			// [mapType]
    long 	*suppressMasks[MAX_TYPES];		// [mapType][scaleI]
    char 	enforce_suppression[MAX_TYPES];		// boolean [mapType]
    char 	optional[MAX_TYPES];			// boolean [mapType]
    MapFeature	default_feature;
    MapStruct	maps[MAX_MAPS];
    long	fatten_thresh;		// widen lines when muPerDot < fatten_thresh
    /* The following are used internally */
    int		maxMaps, maxTypes, maxScales, maxColors;
    char	*mapDir;		// state for read_profile
    int		mapType;		// state for read_profile
    colormap_t	colors;
    unsigned char patterns[MAX_COLORS], useColorPat[MAX_COLORS];
    int		gridWidth, gridColor;	// graticule width and color index
    int		bgColor;		// background color index
    int		highlight_color;	// highlight color index
    float	minAuto, autoRes[MAX_MAPS], minRes[MAX_MAPS];
    int		minAutoI;
    long	minLatSec, maxLatSec, minLngSec, maxLngSec;	// mapset region
    float	minLng, minLat, maxLng, maxLat;		// mapset region in degrees
    struct 	segdict *dictbuf;
    long	dictbuf_len, max_segcount;		// The following are reset by map_setup
    long	canvasNSec, halfWidSec, cntrLngSec;	// coordinates in seconds
    long 	msecPerDot;				// magnified seconds per dot
} MapSet;

@interface CIAMap:NSObject
{
    NSMutableDictionary	*listDict;
    NSPoint		ll, ur, mp;
    MRect		region;
    float		azimutRotation;	// rotation of map with azimut projection
    MapSet		*ms;
    float		gridSize;
    int			drawZodiac, drawDestiny;
    NSDictionary	*mapDict;	// dictionary of displayed map document
    NSColor		*waterColor;
}

+ (CIAMap*)map;

- (void)addSegments:(SegList*)seglist :(int)ptCount;
- (NSMutableDictionary*)generateMapListWithCenter:(NSPoint)center size:(NSSize)size rotation:(float)deg0
                                             grid:(float)grid projection:(char)projectionType
                                                 :(BOOL)zodiacLines :(BOOL)destinyLines
                                           mapset:(NSString*)mapset;
- (int)convertScreenPointToLng:(float*)lng :(float*)lat :(NSPoint)pt;
- (float)mercatorLatitude:(float)latitude;

- (NSPoint)origin;
- (double)scale;

@end

#endif // VHF_H_CIAMAP
