/* Ephemeris.h
 * front end object for the Swiss Ephemeris
 *
 * Copyright (C) 1999-2008 by Georg Fleischmann
 * Author:   Georg Fleischmann
 *
 * created:  1999-11-30
 * modified: 2008-08-23 (EPH_SYMBOLS added)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 *
 * vhf interservice GmbH, Im Marxle 3, 72119 Altingen, Germany
 * eMail: info@vhf.de
 * http://www.vhf.de
 */

#ifndef VHF_H_EPHEMERIS
#define VHF_H_EPHEMERIS

#include	"swephexp.h"

/* planet names and intersections in order of objects array
 * (see swephexp.h: SE_XXX)
 */
#define OBJECTS     [NSArray arrayWithObjects:@"Sun", @"Moon", @"Mercury", @"Venus", @"Mars", @"Jupiter", @"Saturn", @"Uranus", @"Neptune", @"Pluto", @"Node", @"True Node", @"Apogee", @"True Apogee", @"Earth", @"Chiron", @"Pholus", @"Ceres", @"Pallas", @"Juno", @"Vesta", nil]
#define EPH_SYMBOLS [NSArray arrayWithObjects:@"Sun", @"Moo", @"Mer", @"Ven", @"Mar", @"Jup", @"Sat", @"Ura", @"Nep", @"Plu", @"Nod", @"TNo", @"Apo", @"TAp", @"Ear", @"Chi", @"Pho", @"Cer", @"Pal", @"Jun", @"Ves", nil]

#define DIR_NORMAL	0
#define DIR_REVERSE	1

/* 1:1 from SE_XXX + AC, MC */
typedef enum
{
    EPH_SUN         = SE_SUN,
    SUN             = SE_SUN,        // deprecated
    EPH_MOON        = SE_MOON,
    MOON            = SE_MOON,       // deprecated
    EPH_MERCURY     = SE_MERCURY,
    EPH_VENUS       = SE_VENUS,
    EPH_MARS        = SE_MARS,
    EPH_JUPITER     = SE_JUPITER,
    EPH_SATURN      = SE_SATURN,
    EPH_URANUS      = SE_URANUS,
    EPH_NEPTUNE     = SE_NEPTUNE,
    EPH_PLUTO       = SE_PLUTO,
    EPH_NODE        = SE_MEAN_NODE,
    EPH_TRUENODE    = SE_TRUE_NODE,
    EPH_APOGEE      = SE_MEAN_APOG,
    EPH_TRUEAPOGEE  = SE_OSCU_APOG,
    EPH_CHIRON      = SE_CHIRON,
    EPH_PHOLUS      = SE_PHOLUS,
    EPH_CERES       = SE_CERES,
    EPH_PALLAS      = SE_PALLAS,
    EPH_JUNO        = SE_JUNO,
    EPH_VESTA       = SE_VESTA,
    EPH_AC          = 21,
    AC              = 21,            // deprecated
    EPH_MC          = 22,
    MC              = 22,            // deprecated
    EPH_COUNT       = 23,
    EPH_NONE        = -1 // no object
} EphemerisObjects;

/* deprecated */
#define O_SUN		SE_SUN
#define O_MOON		SE_MOON
#define O_MERCURY	SE_MERCURY
#define O_VENUS		SE_VENUS
#define O_MARS		SE_MARS
#define O_JUPITER	SE_JUPITER
#define O_SATURN	SE_SATURN
#define O_URANUS	SE_URANUS
#define O_NEPTUNE	SE_NEPTUNE
#define O_PLUTO		SE_PLUTO
#define O_NODE		SE_MEAN_NODE
#define O_TRUENODE	SE_TRUE_NODE
#define O_APOGEE	SE_MEAN_APOG
#define O_TRUEAPOGEE	SE_OSCU_APOG
#define O_CHIRON	SE_CHIRON
#define O_PHOLUS	SE_PHOLUS
#define O_CERES		SE_CERES
#define O_PALLAS	SE_PALLAS
#define O_JUNO		SE_JUNO
#define O_VESTA		SE_VESTA
#define O_AC		21
#define O_MC		22

#define O_ARRAYSIZE	23	// enough room to store all objects

/* flags */
typedef enum
{
    EPHFLAG_LON =   1,	// ecliptiocal longitude (+ latitude)
    EPHFLAG_LAT =   2,	// ecliptical latitude
    EPHFLAG_DEC =   4,	// declination (+ rectaascension)
    EPHFLAG_REC	=   8,	// rect ascension
    EPHFLAG_NOD	=  16,	// Nodes Lon + Lat (+ Perihelion + Aphelion)
    EPHFLAG_HEL	=  32,	// Perihelion / Aphelion
    //EPHFALG_EQU =  64,	// Equatorial: Node declination, Perigee, Apogee
    EPHFLAG_LOC = 128	// AC / MC
} EphemerisFlags;

/* structure holding the data of each planet
 */
typedef struct
{
    double	lon;		// the zodiac position (ecliptical longitude)
    double	lat;		// Ecliptic latitude
    double	vLon;		// speed in longitude (deg/day), negative = retrograde
    double	vLat;		// speed in latitude  (deg/day), negative = retrograde

    double	rec;		// equatorial rectascension
    double	dec;		// equatorial declination (earth latitude)
    double	vRec;		// speed in rectascension
    double	vDec;		// speed in declination

    double	nodeLon;	// ecliptical longitude of ascending node
    double	nodeLat;	// ecliptical latitude  of ascending node
    double	nodeVLon;	// speed of ascending node in longitude
    double	nodeVLat;	// speed of ascending node in latitude
    double	dscNodeLon;	// ecliptical longitude of descending node
    double	dscNodeLat;	// ecliptical latitude  of descending node
    double	dscNodeVLon;	// speed of descending node in longitude
    double	dscNodeVLat;	// speed of descending node in latitude

    double	perihelionLon;	// ecliptical longitude of perihelion (closest to Sun)
    double	perihelionLat;	// ecliptical latitude  of perihelion
    double	perihelionVLon;	// speed of perihelion in longitude
    double	perihelionVLat;	// speed of perihelion in latitude
    double	aphelionLon;	// ecliptical longitude of aphelion (farthest to Sun)
    double	aphelionLat;	// ecliptical latitude  of aphelion
    double	aphelionVLon;	// speed of aphelion in longitude
    double	aphelionVLat;	// speed of aphelion in latitude

    //double	perigeeLon;	// ecliptical longitude of pregigee (closest to Earth)
    //double	perigeeLat;	// ecliptical latitude  of pregigee
    //double	perigeeVLon;	// speed of pregigee in longitude
    //double	perigeeVLat;	// speed of pregigee in latitude
    //double	apogeeLon;	// ecliptical longitude of apogee (farthest to Earth)
    //double	apogeeLat;	// ecliptical latitude  of apogee
    //double	apogeeVLon;	// speed of apogee in longitude
    //double	apogeeVLat;	// speed of apogee in latitude
}EphObject;

@interface Ephemeris:NSObject
{
    NSArray	*objectNames;
    EphObject	objects[30];
    BOOL	calculateMeanNodes;

    double	calcJD;		// last calculation date
    BOOL	calcTopo;
    float	calcLat, calcLon, calcElev;
}

- (id)init;
- (void)setEphemerisPath:(NSString*)path;
- (NSArray*)objectNames;
- (void)setMeanNodes:(BOOL)flag;

- (int)indexForObjectName:(NSString*)name;
+ (EphemerisObjects)indexForSymbol:(NSString*)symbol;
+ (NSString*)symbolAtIndex:(EphemerisObjects)objIx;

/* date -> julian day (reference = -4712) */
- (double)julianDayForDate:(NSCalendarDate*)utc bc:(BOOL)bcFlag;
- (double)julianDayForYear:(int)year month:(int)month day:(int)day
                      hour:(int)hour minute:(int)minute second:(int)second bc:(BOOL)bcFlag;

/* NSCalendar date methods */
- (EphObject*)objectsAtUTC:(NSCalendarDate*)utc;
- (EphObject*)objectsAtUTC:(NSCalendarDate*)utc lat:(float)lat  lon:(float)lon;
- (EphObject*)objectsAtUTC:(NSCalendarDate*)utc lat:(double)lat lon:(double)lon
                      elev:(double)alt topocentric:(BOOL)topocentric;
- (BOOL)calculatedAtDate:(NSCalendarDate*)date tolerance:(int)tol
                     lat:(float)lat lon:(float)lon elev:(float)elev topocentric:(BOOL)topo;
/* year, month, day, ... methods */
- (EphObject*)objectsAtYear:(int)year month:(int)month day:(int)day
                       hour:(int)hour minute:(int)minute second:(int)second
                  longitude:(double)lon latitude:(double)lat elevation:(double)alt;
/* basic method using julian day (universal time, reference = -4712)
 * also uses external flag: calculateMeanNodes
 */
- (EphObject*)objectsAtJD:(double)tjd flags:(EphemerisFlags)ephFlags
                  objects:(EphemerisObjects*)objs cnt:(int)objCnt
                longitude:(double)lon latitude:(double)lat elevation:(double)alt
              topocentric:(BOOL)topoFlag;

/* dictionary frontend methods */
- (NSMutableDictionary*)objectDictAtUTC:(NSCalendarDate*)utc;
- (NSMutableDictionary*)objectDictAtUTC:(NSCalendarDate*)utc lat:(float)lat lon:(float)lon;
- (NSMutableDictionary*)objectDictAtUTC:(NSCalendarDate*)utc lat:(double)lat lon:(double)lon
                                   elev:(double)alt topocentric:(BOOL)topocentric;
- (NSMutableDictionary*)currentObjectDict;
- (NSMutableDictionary*)currentSpeedLonDict;	// V lon
- (NSMutableDictionary*)currentNodeDict;
- (NSMutableDictionary*)currentDeclinationDict;	// dec
- (NSMutableDictionary*)currentSpeedDecDict;	// V dec

/* houses */
- (double*)houseAtUTC:(NSCalendarDate*)utc latitude:(float)lat longitude:(float)lon houseCount:(int)houseCnt;

@end

#endif	// VHF_H_EPHEMERIS
