/* AstroPrincipal.m
 * Commander of the Astrology module
 *
 * Copyright (C) 2003-2010 by Georg Fleischmann
 * Author:   Georg Fleischmann
 *
 * created:  2003-03-29
 * modified: 2010-05-22
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 *
 * vhf interservice GmbH, Im Marxle 3, 72119 Altingen, Germany
 * eMail: info@vhf.de
 * http://www.vhf.de
 */

#include <AppKit/AppKit.h>
#include <VHFShared/VHFSystemAdditions.h>
#include "../Cenon/App.h"
#include "../Cenon/LayerObject.h"
#include "../Cenon/Graphics.h"
#include "../Cenon/functions.h"
#include "../Cenon/messages.h"
#include "AstroPrincipal.h"
#include "DocViewAstro.h"
#include "DocViewMap.h"
#include "AstroPanels.h"
#include "AstroChart.h"
#include "Astro.bproj/AstroController.h"
#include "CityManager.h"
#include "EventManager.h"
#include "TimeZoneManager.h"
#include "Map.h"
#include "astroMessages.h"

@interface AstroPrincipal(PrivateMethods)
@end

@implementation AstroPrincipal

/*
 * created:  1999-03-15
 * modified: 2006-04-27 (call registration in preferences module)
 * register the defaults
 */
+ (void)initialize
{
    //[AstroController registerDefaults];	// not loaded yet
    NSMutableDictionary	*registrationDict = [NSMutableDictionary dictionary];
    NSArray		*objects;

    [registrationDict setObject:vhfStringFromRGBColor([NSColor whiteColor]) forKey:@"astroBGColor"];
    [registrationDict setObject:@"YES" forKey:@"astroShowPanel"];
    [registrationDict setObject:@"YES" forKey:@"astroShowRetrograde"];
    [registrationDict setObject:@"%d.%m.%Y" forKey:@"astroDateFormat"];
    [registrationDict setObject:@"vhfAstro" forKey:@"astroFont"];
    //[registrationDict setObject:@"YES" forKey:@"astroShowHouseSize"];
    //[registrationDict setObject:@"YES" forKey:@"astroShowHouseMaxima"];
    [registrationDict setObject:@"YES" forKey:@"astroTopocentricPosition"];
    [registrationDict setObject:@"NO"  forKey:@"astroShowDeclination"];
    [registrationDict setObject:@"NO"  forKey:@"astroShowAspectGeo"];
    [registrationDict setObject:@"NO"  forKey:@"astroACFix"];
    [registrationDict setObject:@"0"   forKey:@"astroInterval"];
    objects = [NSArray arrayWithObjects:@"Sun", @"Moon", @"Mercury", @"Venus", @"Mars", @"Jupiter", @"Saturn", @"Uranus", @"Neptune", @"Pluto", @"Chiron", @"True Node", @"True Apogee", nil];
    [registrationDict setObject:objects forKey:@"astroObjects"];

    [[NSUserDefaults standardUserDefaults] registerDefaults:registrationDict];
}

+ (id)instance
{   static id instance = nil;

    if (!instance)
    {   instance = [self alloc];
        [instance init];
    }
    return instance;
}

- (NSString*)version
{   NSDictionary    *infoDict = [[NSBundle mainBundle] infoDictionary];
    NSString        *version = [infoDict objectForKey:@"CFBundleVersion"];

    if ( ! version )
        version = [infoDict objectForKey:@"NSVersion"];     // GNUstep
    return version;
}
- (NSString*)compileDate
{   char    *compileDate = __DATE__;    // Apr 10 2010
    char    date[15];

    if ( strlen(compileDate) == 11 )    // Apr 10 2010 -> 2010-04-10
    {   NSArray *mArray;
        char    mStr[4];
        int     m;

        strncpy(mStr, compileDate, 3); mStr[3] = 0;
        mArray = [NSArray arrayWithObjects:@"Jan", @"Feb", @"Mar", @"Apr", @"May", @"Jun", @"Jul", @"Aug", @"Sep", @"Oct", @"Nov", @"Dec", nil];
        if ( (m = [mArray indexOfObject:[NSString stringWithUTF8String:mStr]]) != NSNotFound )
        {   m ++;
            strncpy(date, compileDate+7, 4);                            // YYYY
            date[4] = '-';
            date[5] = (m >= 10) ? '1' : '0';                            // MM
            if (m >= 10) m -= 10;
            date[6] = '0' + m;
            date[7] = '-';
            date[8] = (compileDate[4] == ' ') ? '0' : compileDate[4];   // DD
            date[9] = compileDate[5];
            date[10] = 0;
            compileDate = date;
        }
    }
    if ( compileDate )
        return [NSString stringWithCString:compileDate encoding:NSASCIIStringEncoding];
    else
        return nil;
}
- (NSString*)serialNo
{
    return nil;
}
- (NSString*)netId
{
    return nil;
}


/*  137.1234 ->  137  7'24"
 * -137.1234 -> -137  7'24"
 */
NSString *stringFromDeg(float deg)
{
    return [AstroPrincipal stringFromDeg:deg];
}
+ (NSString*)stringFromDeg:(float)deg
{   int		d, m, s;
    NSString	*string;

    //if (deg < 0)
    //    NSLog(@"stringFromDeg(): does not properly support negative degrees !");
    d = (int)deg;
    m = (int)floor((deg-d) * 60.0);
    s = (int)floor((deg-d-m/60.0) * 3600.0 + 0.5);
    string = [NSString stringWithFormat:DEGREE_FORMAT,
                       [[NSString stringWithFormat:@"%d", d] stringWithLength:(deg < 0.0) ? -4 : -3],
                       [[NSString stringWithFormat:@"%d", m] stringWithLength:-2],
                       [[NSString stringWithFormat:@"%d", s] stringWithLength:-2] ];
    return string;
}


/* init
 * modified: 2006-07-19
 */
- init
{   NSNotificationCenter	*notificationCenter = [NSNotificationCenter defaultCenter];
    NSFileManager		*fileManager = [NSFileManager defaultManager];
#if !defined(GNUSTEP_BASE_VERSION) && !defined(__APPLE__)	// OpenStep 4.2
    NSMenu			*toolMenu = [[[NSApp mainMenu] itemWithTag:MENU_TOOLS] target];
#else	// Apple, GNUstep
    NSMenu			*toolMenu = [[[NSApp mainMenu] itemAtIndex:MENU_TOOLS-1] submenu];
    NSMenu			*helpMenu = [[[NSApp mainMenu] itemAtIndex:MENU_HELP -1] submenu];
#endif

    /* create HOME-Library entries, if they doesn't exist */
    if ( ![fileManager fileExistsAtPath:localLibrary()] )	// no Cenon master library!
        NSRunAlertPanel(@"", CANTFINDLIB_STRING, OK_STRING, nil, nil, NULL);
    else
    {   NSString	*path = userLibrary(), *from = nil, *to, *dir = nil, *folder;
        int		i;

        /* create "Cenon/Astro" */
        to = vhfPathWithPathComponents(path, MODULENAME, nil);
        if ( ![fileManager fileExistsAtPath:to] )
            [fileManager createDirectoryAtPath:to attributes:nil];

        /* create folders + copy "Cenon/Astro/.../.dir.tiff" */
        for (i=0; i<3; i++)
        {
            switch (i)
            {
                case 0:
                    from = vhfPathWithPathComponents(localLibrary(), MODULENAME, @".dir.tiff", nil);
                    dir  = vhfPathWithPathComponents(path, MODULENAME, nil);
                    to   = vhfPathWithPathComponents(dir, @".dir.tiff", nil);
                    break;
                case 1: folder = @"Charts";
                case 2: if (i == 2) folder = @"Maps";
                    from = vhfPathWithPathComponents(localLibrary(), MODULENAME, folder, @".dir.tiff", nil);
                    dir  = vhfPathWithPathComponents(path, MODULENAME, folder, nil);
                    to   = vhfPathWithPathComponents(dir, @".dir.tiff", nil);
            }
            if ( dir && ![fileManager fileExistsAtPath:dir] )
                [fileManager createDirectoryAtPath:dir attributes:nil];
            if ( ![fileManager fileExistsAtPath:to] )
                [fileManager copyPath:from toPath:to handler:nil];
        }
    }

    /* add observer for documents that open */
    [notificationCenter addObserver:self
                           selector:@selector(documentDidOpen:)
                               name:DocumentDidOpen
                             object:nil];

    /* add observer to calculate chart */
    [notificationCenter addObserver:self
                           selector:@selector(calculateChart:)
                               name:CalculateChartNotification
                             object:nil];

    /* add observer to load map image at date */
    [notificationCenter addObserver:self
                           selector:@selector(updateMapForDate:)
                               name:AstroDateSetNotification
                             object:nil];

    /* add a menu item for the AstroPanel */
#ifdef __APPLE__
    [toolMenu addItem:[NSMenuItem separatorItem]];
#endif
    [toolMenu addItemWithTitle:@"Astro Panel ..."
                        action:@selector(showAstroPanel:)
                 keyEquivalent:@""];
    [[[toolMenu itemArray] objectAtIndex:[[toolMenu itemArray] count]-1] setTarget:self];
    if (Prefs_ShowAstroPanel)
        [self showAstroPanel:self];

    /* Help Menu */
#ifdef __APPLE__
    {   int i;

        /* redirect menu entries to Astro website */
        for ( i=0; i<[helpMenu numberOfItems]; i++ )
        {   NSMenuItem  *item = [helpMenu itemAtIndex:i];

            if ( [item action] == @selector(showWebPage:) )
                [item setTarget:self];  // redirect to our function (we are not first responder)
        }

        /* add a menu item for our documentation */
        [helpMenu addItemWithTitle:@"Cenon Astro"
                            action:@selector(showAstroHelp:)
                     keyEquivalent:@""];
        [[[helpMenu itemArray] objectAtIndex:[[helpMenu itemArray] count]-1] setTarget:self];
    }
#endif

    /* create event manager, city manager, time zone manager */
    eventManager    = [EventManager new];
    cityManager     = [CityManager new];
    timeZoneManager = [TimeZoneManager new];

    return self;
}



/* AstroPanel stuff
 */
- (void)showAstroPanel:sender
{
    if (!astroPanel)
    {   NSBundle	*bundle = [NSBundle bundleForClass:[AstroPanel class]];

        //if (![NSBundle loadModelNamed:@"AstroPanel" owner:self])
        if ( ![bundle loadNibFile:@"AstroPanel"
                externalNameTable:[NSDictionary dictionaryWithObject:self forKey:@"NSOwner"]
                         withZone:[self zone]] )
            NSLog(@"Cannot load AstroPanel interface file");
        [astroPanel awakeFromNib];  // FIXME: GNustep doesn't call this on it's own
    }
    [astroPanel setFrameUsingName:@"AstroPanel"];
    [astroPanel setFrameAutosaveName:@"AstroPanel"];
    [astroPanel setBecomesKeyOnlyIfNeeded:NO];
/*#ifndef GNUSTEP_BASE_VERSION
    [astroPanel setFloatingPanel:YES];
#endif*/
    [astroPanel orderFront:sender];
}
- astroPanel
{
    return astroPanel;
}

/* Show Web Page
 * created: 2010-05-22
 */
- (void)showWebPage:(id)sender
{   NSString    *site;
    int         tag = [sender tag];

	if ([sender isKindOfClass:[NSMatrix class]])
		tag = [sender selectedTag];
	switch (tag)
    {
        default: site = VHFLocalAstroString(@"http://www.cenon.info/astro"); break;                      // Web Site
        case 1:  site = VHFLocalAstroString(@"http://www.cenon.info/astro/support_faq_gb.html"); break;  // FAQ
        case 2:  site = VHFLocalAstroString(@"http://www.cenon.info/astro/releaseNotes_gb.html"); break; // Release Notes
        case 3:  site = VHFLocalAstroString(@"mailto:service@vhf.de?subject=Cenon%20Astro%20Feedback"); break;  // eMail
        case 4:  site = VHFLocalAstroString(@"http://www.cenon.info/astro/news/news_gb.html"); break;         // Check for Updates
            //site = NSLocalizedString(@"http://www.cenon.biz/download_gb.html", Download); break;
    }
	[[NSWorkspace sharedWorkspace] openURL:[NSURL URLWithString:site]];
}

/* load PDF documentation
 * Note: each module can add a help menu entry to load it's docu
 * modified: 2007-07-19
 */
- (void)showAstroHelp:sender
{
#if defined(GNUSTEP_BASE_VERSION) || defined(__APPLE__)	// GNUstep or Apple
    NSBundle	*bundle = [NSBundle bundleForClass:[AstroPrincipal class]];
    NSArray     *localizations;
    NSString    *locale, *path, *helpFile;
    int         l, i;
    
    localizations = [NSBundle preferredLocalizationsFromArray:[bundle localizations]];
    for (l=0; l<[localizations count]; l++ )
    {
        locale = [[localizations objectAtIndex:l] stringByAppendingString:@".lproj"];
        for (i=0; i<3; i++)
        {
            switch (i)
            {
                case 0: path = [bundle resourcePath];
                    helpFile = vhfPathWithPathComponents(path, locale, DOCU_ASTRO, nil);
                    break;
                case 1: path = vhfUserLibrary(APPNAME);
                    helpFile = vhfPathWithPathComponents(path, @"Docu", locale, DOCU_ASTRO, nil);
                    break;
                case 2: path = vhfLocalLibrary(APPNAME);
                    helpFile = vhfPathWithPathComponents(path, @"Docu", locale, DOCU_ASTRO, nil);
                    break;
                default:
                    return;
            }
            if ( [[NSFileManager defaultManager] fileExistsAtPath:helpFile] )
            {   [[NSWorkspace sharedWorkspace] openFile:helpFile];
                l = [localizations count];
                break;
            }
        }
    }
#endif
}

- eventManager		{ return eventManager; }
- cityManager		{ return cityManager; }
- timeZoneManager	{ return timeZoneManager; }

- (Ephemeris*)ephemeris
{
    if (!ephemeris)
    {   NSFileManager	*fileManager = [NSFileManager defaultManager];
        NSArray		*paths;
        int		i;
        NSString	*path = nil;

        /* set path to ephemeris files */
        paths = [NSArray arrayWithObjects:
                         [[[NSBundle bundleForClass:[Ephemeris class]] resourcePath]
                          stringByAppendingPathComponent:@"Ephemeris"],
                         [userLibrary()  stringByAppendingPathComponent:@"Astro/Ephemeris"],
                         [localLibrary() stringByAppendingPathComponent:@"Astro/Ephemeris"], nil];
        for (i=0; i<(int)[paths count]; i++)
            if ([fileManager fileExistsAtPath:[paths objectAtIndex:i]] || i == (int)[paths count]-1)
            {   path = [paths objectAtIndex:i];
                break;
            }
        ephemeris = [Ephemeris new];
        [ephemeris setEphemerisPath:path];
    }
    return ephemeris;
}


/* notification that a new document has been opened
 * modified: 2010-04-22 (Map fixed)
 *           2010-01-17 (remove scroller for Chart and scale to window size)
 */
- (void)documentDidOpen:(NSNotification*)notification
{   DocView	*view = [notification object];

    if ([view isMemberOfClass:[DocView class]])
    {   NSString    *name = [(Document*)[view document] name];

        /* these methods are called for every view.
         * We have to make sure that we do our special stuff for our documents only.
         */
        if ( [name hasPrefix:@"Map"] )
        {
            [view initMap]; // this loads the map.plist and stores it into [view statusDict]
            if ([view respondsToSelector:@selector(mapDocumentHasBeenSaved:)])
            {   NSNotificationCenter    *notificationCenter = [NSNotificationCenter defaultCenter];
                [notificationCenter addObserver:view
                                       selector:@selector(mapDocumentHasBeenSaved:)
                                           name:DocumentHasBeenSaved
                                         object:nil];
            }
        }

        if ( ! [name hasPrefix:@"Chart"] && ! [name hasPrefix:@"Map"] )
            return; // none of our documents

        /* remove scrollers */
        if ( ! [(DocWindow*)[view window] hasCoordDisplay] )
        {   id  scrollView = [[view document] scrollView];

            [scrollView setHasHorizontalScroller:NO];
            [scrollView setHasVerticalScroller:  NO];

            /* scale to fit window */
            {   NSRect  bounds = [view bounds];
                NSRect  frame  = [[view window] frame];
                float   oldScale = [view scaleFactor];
                float   scale = (frame.size.width  / bounds.size.width) / oldScale;

                [[view document] scale:scale :scale withCenter:NSZeroPoint];
            }
        }
    }
    else
        NSLog(@"AstroPrincipal, notification send from object %@, shoult be DocView!", view);
}

/* notification that we have to calculate a chart (CalculateChartNotification)
 *
 * dict   name	    NSString       title of the chart
 *        date      NSCalendarDate date of the chart
 *        lat       NSNumber       latitude
 *        lon       NSNumber       longitude
 *        city      NSString       location
 *        composite NSDictionary   optional dictionary with entries like above for a composite chart
 *
 * modified: 2009-04-01 (no data for main chart but composite -> we set composite only)
 */
- (void)calculateChart:(NSNotification*)notification
{   NSDictionary	*dict = [notification object];

    if ([dict isKindOfClass:[NSDictionary class]])
    {   int         i;
        Document    *doc;
        AstroChart  *astroChart;

        for ( i=[[NSApp windows] count]-1; i>=0; i-- )
        {   doc = [(App*)NSApp documentInWindow:[[(App*)NSApp windows] objectAtIndex:i]];
            if ( [[doc name] hasPrefix:CHART_PREFIX] )
                break;
        }
        if ( i < 0 )
        {   NSLog(@"AstroPrincipal: No document found with prefix '%@' to chart data!", CHART_PREFIX);
            return;
        }

        astroChart = [[astroPanel windowAt:AP_CHART] astroChart];
        if (!astroChart)
            astroChart = [AstroChart astroChartWithView:[doc documentView]];
#if 0
        /* only composite -> use data from astro panel */
        if ( ![dict objectForKey:@"date"] &&
             [dict objectForKey:@"composite"] && [[dict objectForKey:@"composite"] objectForKey:@"date"] )
        {   //NSDictionary    data = [[astroPanel windowAt:AP_CHART] data];

            //[data setObject:[dict objectForKey:@"composite"] forKey:@"composite"];
            //dict = data;
            dict = [[dict objectForKey:@"composite"] mutableCopy];
            [(NSMutableDictionary*)dict
                    setObject:[[dict objectForKey:@"date"] descriptionWithCalendarFormat:@"%Y-%m-%d %H:%M %z"]
                    forKey:@"date"];   // workaround: we have to fix [APChart setCompositeEvent]
            // this puts the data to the right side of the astro-panel (inside) and will redraw outside + inside
            // what we want is to draw only the inside with the method below: no date -> draw inside only
            [[NSNotificationCenter defaultCenter] postNotificationName:AstroUpdateEvent
                                                                object:dict
                                                              userInfo:[NSDictionary dictionaryWithObject:@"1" forKey:@"composite"]];
        }
        //else
#endif
        {
            [astroChart radix:[dict objectForKey:@"date"]
                    longitude:[dict floatForKey:@"lon"]
                     latitude:[dict floatForKey:@"lat"]
                         city:[dict objectForKey:@"city"]
                        title:[dict objectForKey:@"name"]
                    composite:[dict objectForKey:@"composite"]];
        }
    }
    else
        NSLog(@"AstroPrincipal (calculateChart), notification send with an object not a dictionary!");
}

/* notification that the date was set, and we have to load the corresponding map image
 * object   NSCalendarDate with date and time of map to load
 * modified: 2006-04-03
 */
- (void)updateMapForDate:(NSNotification*)notification
{   NSCalendarDate	*date = [notification object];

    if ([date isKindOfClass:[NSCalendarDate class]])
    {   int         i;
        Document    *doc = [(App*)NSApp currentDocument];
        DocView     *view = [doc documentView];
        Map         *map;

        if ( ![[doc name] hasPrefix:MAP_PREFIX] )
        {
            for ( i=[[NSApp windows] count]-1; i>=0; i-- )
            {   doc = [(App*)NSApp documentInWindow:[[(App*)NSApp windows] objectAtIndex:i]];
                if ( [[doc name] hasPrefix:MAP_PREFIX] )
                    break;
            }
            if ( i < 0 )
                return;
        }

        view = [doc documentView];
        map = [[view statusDict] objectForKey:@"map"];
        [map displayMapAtDate:date inView:view];
    }
    else
        NSLog(@"AstroPrincipal (updateMapForDate), notification send with an object not a date!");
}

@end
