/* Ephemeris.m
 * front end object for the Swiss Ephemeris
 *
 * Copyright (C) 1999-2011 by vhf interservice GmbH
 * Author:   Georg Fleischmann
 *
 * created:  2001-09-02
 * modified: 2011-04-24 (-objectSymbols)
 *           2008-08-23 (-indexForObjectSymbol: added)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 *
 * vhf interservice GmbH, Im Marxle 3, 72119 Altingen, Germany
 * eMail: info@vhf.de
 * http://www.vhf.de
 */

#include <Foundation/Foundation.h>
#include <VHFShared/types.h>
#include "Ephemeris.h"
#include "../astroCommon.h"		// sideralTime(), ac(), mc(), sunNode()

static char *ephepath = NULL;
static NSArray *objectSymbols = nil;

@interface Ephemeris(PrivateMethods)
@end

/*FILE *FileOpen(char *szFile, int nFileMode)
{   FILE	*file;
    char	name[1024];
    NSString	*resourcePath = [[NSBundle bundleForClass:[Ephemeris class]] resourcePath];

    strcpy( name, [resourcePath cString]);
    strcat( name, "/");
    strcat( name, szFile);

    file = fopen(name, "rb");
    if (file == NULL )
        NSLog(@"File '%s' not found.", name);
    return file;
}*/

@implementation Ephemeris

- (id)init
{
    [super init];
    objectNames   = [OBJECTS retain];
    if (!objectSymbols)
        objectSymbols = [EPH_SYMBOLS retain];

    return self;
}

- (void)setEphemerisPath:(NSString*)path
{
    if (path)
    {
        ephepath = calloc([path cStringLength]+1, sizeof(char));
        [path getCString:ephepath];
        swe_set_ephe_path(ephepath);
    }
}

- (NSArray*)objectNames
{
    return objectNames;
}
- (NSArray*)objectSymbols
{
    return objectSymbols;
}

- (void)setMeanNodes:(BOOL)flag
{
    calculateMeanNodes = flag;
}

- (int)indexForObjectName:(NSString*)name
{
    if ([name isEqual:@"AC"])
        return EPH_AC;
    else if ([name isEqual:@"MC"])
        return EPH_MC;
    return [objectNames indexOfObject:name];
}
+ (EphemerisObjects)indexForSymbol:(NSString*)symbol
{
    if ([symbol isEqual:@"AC"])
        return EPH_AC;
    else if ([symbol isEqual:@"MC"])
        return EPH_MC;
    return [objectSymbols indexOfObject:symbol];
}
+ (NSString*)symbolAtIndex:(EphemerisObjects)objIx
{
    switch (objIx)
    {
        case EPH_AC: return @"AC";
        case EPH_MC: return @"MC";
        default:     return [objectSymbols objectAtIndex:objIx];
    }
}

/* get objects including AC and MC
 */
- (EphObject*)objectsAtUTC:(NSCalendarDate*)utc lat:(float)lat lon:(float)lon
{   NSCalendarDate	*st = sideralTime(utc, lon);
    EphObject		*objs = [self objectsAtUTC:utc];

    /* add ac, mc */
    objs[O_AC].lon = ac(st, lat);
    objs[O_AC].vLon = 1.0;
    objs[O_AC].vLat = 0.0;
    objs[O_MC].lon = mc(st);
    objs[O_MC].vLon = 1.0;
    objs[O_MC].vLat = 0.0;

    return objs;
}
- (EphObject*)objectsAtUTC:(NSCalendarDate*)utc lat:(double)lat lon:(double)lon
                      elev:(double)alt topocentric:(BOOL)topocentric
{   NSCalendarDate	*st = sideralTime(utc, lon);
    EphObject		*objs;

    if (!topocentric)
        lon = -360.0;
    objs = [self objectsAtYear:[utc yearOfCommonEra]  month:[utc monthOfYear]     day:[utc dayOfMonth]
                          hour:[utc hourOfDay]       minute:[utc minuteOfHour] second:[utc secondOfMinute]
                     longitude:lon latitude:lat elevation:alt];

    /* add ac, mc */
    objs[O_AC].lon = ac(st, lat);
    objs[O_AC].vLon = 1.0;
    objs[O_AC].vLat = 0.0;
    objs[O_MC].lon = mc(st);
    objs[O_MC].vLon = 1.0;
    objs[O_MC].vLat = 0.0;

    return objs;
}

- (EphObject*)objectsAtUTC:(NSCalendarDate*)utc
{
    /* we have day, month and year and convert to Julian day number */
    //h = (double)[utc hourOfDay] + ((double)[utc minuteOfHour]*60.0 + (double)[utc secondOfMinute]) / 3600.0;
    //tjd = swe_julday([utc yearOfCommonEra], [utc monthOfYear], [utc dayOfMonth], h, SE_GREG_CAL);

    return [self objectsAtYear:[utc yearOfCommonEra]  month:[utc monthOfYear]     day:[utc dayOfMonth]
                          hour:[utc hourOfDay]       minute:[utc minuteOfHour] second:[utc secondOfMinute]
                     longitude:-360.0 latitude:-360.0 elevation:0.0];
}

/* created:  1999
 * modified: 2006-02-02 (Asteroids added)
 *
 * return ephemeris, projection on earth
 * the method returns a dictionary containing the following values for each planet:
 * angle, declination, direction
 *
 * Parameters:
 *  longitude:	E = +  W = -
 *  latitude:	N = +  S = -
 *  elevation:	meters above sea level
 *
 * dictionary =
 * {
 *     "gmt" = NSCalendarDate;
 *     "Moon" = (angle, declination, DIR_NORMAL);
 *     "Sun" = (12.2, 0.0, DIR_NORMAL);
 *     ...
 * }
 */
- (EphObject*)objectsAtYear:(int)year  month:(int)month     day:(int)day
                       hour:(int)hour minute:(int)minute second:(int)second
                  longitude:(double)lon latitude:(double)lat elevation:(double)alt
{   int		p;
    double	tjd, x2[6], h;
    long	iflag = SEFLG_SPEED, iflgret, topoFlag = 0;
    char	serr[AS_MAXCH];

    /* we have day, month and year and convert to Julian day number */
    //h = (double)[utc hourOfDay] + ((double)[utc minuteOfHour]*60.0 + (double)[utc secondOfMinute]) / 3600.0;
    //tjd = swe_julday([utc yearOfCommonEra], [utc monthOfYear], [utc dayOfMonth], h, SE_GREG_CAL);
    h = (double)hour + (double)(minute*60 + second) / 3600.0;
    tjd = swe_julday(year, month, day, h, SE_GREG_CAL);
    /* compute Ephemeris time from Universal time by adding delta_t */
    //te = tjd + swe_deltat(tjd);

    /* topocentric positions (observed from location on earth) */
    if (lon >= -180.0 && lat >= -90.0)
    {
        swe_set_topo(lon, lat, alt);
        topoFlag = SEFLG_TOPOCTR;
    }

    /* a loop over all planets */
    for (p = SE_SUN; p <= SE_VESTA; p++)
    {
        if (p == SE_EARTH)
            continue;

        /* do the coordinate calculation for this planet p
         * x[0] = longitudinal position on ecliptic
         * x[1] = latitude position on ecliptic
         * x[2] = distance in AU
         * x[3] = speed in longitude (deg/day), negative = retrograde
         * x[4] = speed in latitude (deg/day), negative = retrograde
         * x[5] = speed in distance (AU/day)
         */
        iflag = SEFLG_SPEED | topoFlag;
        if ((iflgret = swe_calc_ut(tjd, p, iflag, x2, serr)) < 0)
            NSLog(@"error: %s\n", serr);
        else if (iflgret != iflag)
            NSLog(@"warning: iflgret != iflag.\n");
        objects[p].lon  = x2[0];	// longitude
        objects[p].lat  = x2[1];	// latitude
        //objects[p].dist = x2[2];	// distance in AU
        objects[p].vLon = x2[3];	// speed longitude
        objects[p].vLat = x2[4];	// speed latitude
        //objects[p].vDist = x2[5];	// speed distance

        /* equatorial (SEFLG_EQUATORIAL) */
        if ((iflgret = swe_calc_ut(tjd, p, iflag | SEFLG_EQUATORIAL, x2, serr)) < 0)
            NSLog(@"error: %s (equatorial).\n", serr);
        else if (iflgret != (iflag|SEFLG_EQUATORIAL))
            NSLog(@"warning: iflgret (%f) != iflag (%f) (equatorial).\n", iflgret, iflag|SEFLG_EQUATORIAL);
        objects[p].rec  = x2[0];	// rectascension
        objects[p].dec  = x2[1];	// declination
        //objects[p].dist = x2[2];	// distance in AU
        objects[p].vRec = x2[3];	// speed rectascension
        objects[p].vDec = x2[4];	// speed declination
        //objects[p].vDist = x2[5];	// speed distance

        /* planetary nodes
         * SE_NODBIT_MEAN = mean nodes
         * SE_NODBIT_OSCU = osculating nodes
         * SE_NODBIT_OSCU_BAR = beyond Jupiter with barycentric ellipses
         * SE_NODBIT_OSCU_BAR + SE_NODBIT_MEAN = mean values up to Neptune
         * SE_NODBIT_FOPOINT = second focal point is returned in xaphe
         *
         * xnasc = ascending node
         * xndsc = descending node
         * xperi = prihelion
         * xaphe = aphelion
         */

        if (p >= SE_MEAN_NODE && p <= SE_OSCU_APOG)	// skip Moon node/apogee
            continue;

        iflag = SEFLG_SPEED /*| topoFlag*/;
        {   double	xnasc[6], xndsc[6], xperi[6], xaphe[6];
            int		nodeBit = (calculateMeanNodes) ? SE_NODBIT_MEAN : SE_NODBIT_OSCU;

            if ((iflgret = swe_nod_aps_ut(tjd, p, iflag, nodeBit, xnasc, xndsc, xperi, xaphe, serr)) < 0)
                NSLog(@"error: %s (Nodes)\n", serr);
            //else if (iflgret != iflag)
            //    NSLog(@"warning: iflgret != iflag (Nodes) iflgret = %d.\n", iflgret);
            /* (asc/dsc nodes = 0 for Sun) */
            objects[p].nodeLon     = xnasc[0];		// longitude
            objects[p].nodeLat     = xnasc[1];		// latitude (0 for Moon)
            //objects[p].nodeDist    = xnasc[2];	// distance in AU
            objects[p].nodeVLon    = xnasc[3];		// speed longitude
            objects[p].nodeVLat    = xnasc[4];		// speed latitude (0 for Moon)
            //objects[p].nodeVDist   = xnasc[5];	// speed distance

            objects[p].dscNodeLon  = xndsc[0];		// (asc + 180 deg for Moon)
            objects[p].dscNodeLat  = xndsc[1];		// (0 for Moon)
            //objects[p].dscNodeDist = xndsc[2];	// distance in AU
            objects[p].dscNodeVLon = xndsc[3];		// (asc + 0 for Moon and Sun)
            objects[p].dscNodeVLat = xndsc[4];		// (0 for Moon)
            //objects[p].dscNodeVDist = xndsc[5];

            objects[p].perihelionLon  = xperi[0];
            objects[p].perihelionLat  = xperi[1];
            objects[p].perihelionVLon = xperi[3];
            objects[p].perihelionVLat = xperi[4];
            objects[p].aphelionLon    = xaphe[0];	// (peri + 180 for Sun and Moon)
            objects[p].aphelionLat    = xaphe[1];	// (-peri      for Sun and Moon)
            objects[p].aphelionVLon   = xaphe[3];	// (peri + 0   for Sun and Moon)
            objects[p].aphelionVLat   = xaphe[4];	// (-peri      for Sun and Moon)

            /*{   char	s[40];
                swe_get_planet_name(p, s);
                NSLog(@"%10s asc=%f %f %f %f %f %f", s, xnasc[0], xnasc[1], xnasc[2], xnasc[3], xnasc[4], xnasc[5]);
                NSLog(@"%10s dsc=%f %f %f %f %f %f", s, xndsc[0], xndsc[1], xndsc[2], xndsc[3], xndsc[4], xndsc[5]);
                NSLog(@"%10s peri=%f %f %f %f %f %f", s, xperi[0], xperi[1], xperi[2], xperi[3], xperi[4], xperi[5]);
                NSLog(@"%10s aphe=%f %f %f %f %f %f", s, xaphe[0], xaphe[1], xaphe[2], xaphe[3], xaphe[4], xaphe[5]);
            }*/
#if 0
            /* equatorial (SEFLG_EQUATORIAL) */
            if ( (iflgret = swe_nod_aps_ut(tjd, p, iflag|SEFLG_EQUATORIAL,
                                           nodeBit, xnasc, xndsc, xperi, xaphe, serr)) < 0 )
                NSLog(@"error: %s (Nodes equatorial).\n", serr);
            /*else if (iflgret != (iflag|SEFLG_EQUATORIAL))
                NSLog(@"warning: iflgret (%f) != iflag (%f) (Nodes equatorial).\n",
                      iflgret, iflag|SEFLG_EQUATORIAL);*/
            //objects[p].nodeRec     = xnasc[0];	// rectascension
            objects[p].nodeDec     = xnasc[1];		// declination
            //objects[p].nodeDist    = xnasc[2];	// distance in AU
            //objects[p].nodeVRec    = xnasc[3];	// speed rectascension
            objects[p].nodeVDec    = xnasc[4];		// speed declination
            //objects[p].nodeVDist   = xnasc[5];	// speed distance
            //objects[p].dscNodeRec  = xndsc[0];	
            objects[p].dscNodeDec  = xndsc[1];
            //objects[p].dscNodeDist = xndsc[2];
            //objects[p].dscNodeVRec = xndsc[3];
            objects[p].dscNodeVDec = xndsc[4];
            //objects[p].dscNodeVDist = xndsc[5];

            objects[p].perigeeLon   = xperi[0];
            objects[p].perigeeLat   = xperi[1];
            objects[p].perigeeVLon  = xperi[3];
            objects[p].perigeeVLat  = xperi[4];
            objects[p].apogeeLon    = xaphe[0];
            objects[p].apogeeLat    = xaphe[1];
            objects[p].apogeeVLon   = xaphe[3];
            objects[p].apogeeVLat   = xaphe[4];
#endif
        }

        /* get the name of the planet p, debugging */
        /*{   char	snam[40];

            swe_get_planet_name(p, snam);
            printf("%10s\t%11.7f\t%10.7f\t%10.7f\t%10.7f\n", snam, x2[0], x2[1], x2[2], x2[3]);
        }*/
    }

    /* sun node */
    objects[SE_SUN].nodeLon = sunNode(year);
    objects[SE_SUN].nodeLat = 0.0;

    return objects;
}

/*
 * convert to julian date
 * utc      1 AD historical year -> year  1 (astronomical)
 *          0 -> does not exist
 *          1 BC historical year -> year  0 (astronomical)
 *          2 BC historical year -> year -1 (astronomical)
 * bcFlag   YES = before christ (utc = >= 1)
 *          NO  = ano domini    (utc = >= 1 ... or <= -1)
 */
- (double)julianDayForDate:(NSCalendarDate*)utc bc:(BOOL)bcFlag
{   int		year, month, day;
    double	hour, tjd;

    year  = [utc yearOfCommonEra];
    if (bcFlag && year != 0)
        year = -year + 1;
    month = [utc monthOfYear];
    day   = [utc dayOfMonth];
    hour  = (double)[utc hourOfDay] + (double)([utc minuteOfHour]*60 + [utc secondOfMinute]) / 3600.0;
    tjd = swe_julday(year, month, day, hour, SE_GREG_CAL);

/*printf("%f (%f) = %f (%f)\n", tjd, tjd + swe_deltat(tjd), vhfJulianDay(utc), mjd(utc));
hour = [sideralTime(utc, 9.48) timeIntervalSinceDate:[NSCalendarDate dateWithString:[sideralTime(utc, 9.48) descriptionWithCalendarFormat:@"%Y.%m.%d-00:00 %z"] calendarFormat:@"%Y.%m.%d-%H:%M %z"]]/3600.0;
printf("st = %f = %f\n", vhfSideralTime(tjd, 9.48), hour);*/
    return tjd;
}
- (double)julianDayForYear:(int)year month:(int)month day:(int)day
                      hour:(int)h minute:(int)m second:(int)s bc:(BOOL)bcFlag
{   double	hour, tjd;

    if (bcFlag && year > 0)
        year = -year + 1;
    hour = (double)h + (double)(m*60 + s) / 3600.0;
    tjd = swe_julday(year, month, day, hour, SE_GREG_CAL);
    return tjd;
}

/* Basic ephemeris calculation
 * created:  2006-02-02 (replacement of other method)
 * modified: 2006-02-02
 *
 * Parameters:
 *  mjd:         modified julian day
 *  objects:     array with objects
 *  objCnt:      number of items in objects[]
 *  longitude:	 E = +  W = -
 *  latitude:	 N = +  S = -
 *  elevation:	 meters above sea level
 *  topocentric: wheather we do topocentric versus geocentric calculation
 *
 * External flags:
 *  calculateMeanNodes
 */
- (EphObject*)objectsAtJD:(double)tjd flags:(EphemerisFlags)ephFlags
                  objects:(EphemerisObjects*)objs cnt:(int)objCnt
                longitude:(double)lon latitude:(double)lat elevation:(double)alt
              topocentric:(BOOL)topoFlag
{   int		i, p;
    double	x2[6];
    long	iflag = SEFLG_SPEED, iflgret;
    char	serr[AS_MAXCH];
    int		nodeBit = (calculateMeanNodes) ? SE_NODBIT_MEAN : SE_NODBIT_OSCU;

    /* compute Ephemeris time from Universal time by adding delta_t */
    //te = tjd + swe_deltat(tjd);

    /* topocentric positions (observed from location on earth) */
    if (topoFlag && lon >= -180.0 && lat >= -90.0)
        swe_set_topo(lon, lat, alt);
    else
        topoFlag = NO;

    /* a loop over all planets */
    for (i = 0; i < objCnt; i++)
    {
        p = objs[i];
        if (p == SE_EARTH || p == EPH_AC || p == EPH_MC)
            continue;

        /* do the coordinate calculation for this planet p
         * x[0] = longitudinal position on ecliptic
         * x[1] = latitude position on ecliptic
         * x[2] = distance in AU
         * x[3] = speed in longitude (deg/day), negative = retrograde
         * x[4] = speed in latitude (deg/day), negative = retrograde
         * x[5] = speed in distance (AU/day)
         */

        iflag = SEFLG_SPEED | ((topoFlag) ? SEFLG_TOPOCTR : 0);

        /* ecliptical */
        if ( ephFlags & (EPHFLAG_LON + EPHFLAG_LAT))
        {
            if ((iflgret = swe_calc_ut(tjd, p, iflag, x2, serr)) < 0)
                NSLog(@"error: %s\n", serr);
            else if (iflgret != iflag)
                NSLog(@"warning: iflgret != iflag.\n");
            objects[p].lon  = x2[0];	// longitude
            objects[p].lat  = x2[1];	// latitude
            //objects[p].dist = x2[2];	// distance in AU
            objects[p].vLon = x2[3];	// speed longitude
            objects[p].vLat = x2[4];	// speed latitude
            //objects[p].vDist = x2[5];	// speed distance
        }
        else
            objects[p].lon = objects[p].lat = objects[p].vLon = objects[p].vLat = 0.0;

        /* equatorial (SEFLG_EQUATORIAL) */
        if ( ephFlags & (EPHFLAG_DEC + EPHFLAG_REC))
        {
            if ((iflgret = swe_calc_ut(tjd, p, iflag | SEFLG_EQUATORIAL, x2, serr)) < 0)
                NSLog(@"error: %s (equatorial).\n", serr);
            else if (iflgret != (iflag|SEFLG_EQUATORIAL))
                NSLog(@"warning: iflgret (%f) != iflag (%f) (equatorial).\n", iflgret, iflag|SEFLG_EQUATORIAL);
            objects[p].rec  = x2[0];	// rectascension
            objects[p].dec  = x2[1];	// declination
            //objects[p].dist = x2[2];	// distance in AU
            objects[p].vRec = x2[3];	// speed rectascension
            objects[p].vDec = x2[4];	// speed declination
            //objects[p].vDist = x2[5];	// speed distance
        }
        else
            objects[p].rec = objects[p].dec = objects[p].vRec = objects[p].vDec = 0.0;

        /* planetary nodes + perhelion/aphelion
         * SE_NODBIT_MEAN = mean nodes
         * SE_NODBIT_OSCU = osculating nodes
         * SE_NODBIT_OSCU_BAR = beyond Jupiter with barycentric ellipses
         * SE_NODBIT_OSCU_BAR + SE_NODBIT_MEAN = mean values up to Neptune
         * SE_NODBIT_FOPOINT = second focal point is returned in xaphe
         *
         * xnasc = ascending node
         * xndsc = descending node
         * xperi = prihelion
         * xaphe = aphelion
         */
        if (ephFlags & EPHFLAG_NOD)
        {   double	xnasc[6], xndsc[6], xperi[6], xaphe[6];

            if (p >= SE_MEAN_NODE && p <= SE_OSCU_APOG)	// skip Moon node/apogee
                continue;

            iflag = SEFLG_SPEED /*| ((topoFlag) ? SEFLG_TOPOCTR : 0)*/;

            if ((iflgret = swe_nod_aps_ut(tjd, p, iflag, nodeBit, xnasc, xndsc, xperi, xaphe, serr)) < 0)
                NSLog(@"error: %s (Nodes)\n", serr);
            //else if (iflgret != iflag)
            //    NSLog(@"warning: iflgret != iflag (Nodes) iflgret = %d.\n", iflgret);
            /* (asc/dsc nodes = 0 for Sun) */
            objects[p].nodeLon     = xnasc[0];		// longitude
            objects[p].nodeLat     = xnasc[1];		// latitude (0 for Moon)
            //objects[p].nodeDist    = xnasc[2];	// distance in AU
            objects[p].nodeVLon    = xnasc[3];		// speed longitude
            objects[p].nodeVLat    = xnasc[4];		// speed latitude (0 for Moon)
            //objects[p].nodeVDist   = xnasc[5];	// speed distance

            objects[p].dscNodeLon  = xndsc[0];		// (asc + 180 deg for Moon)
            objects[p].dscNodeLat  = xndsc[1];		// (0 for Moon)
            //objects[p].dscNodeDist = xndsc[2];	// distance in AU
            objects[p].dscNodeVLon = xndsc[3];		// (asc + 0 for Moon and Sun)
            objects[p].dscNodeVLat = xndsc[4];		// (0 for Moon)
            //objects[p].dscNodeVDist = xndsc[5];

            objects[p].perihelionLon  = xperi[0];
            objects[p].perihelionLat  = xperi[1];
            objects[p].perihelionVLon = xperi[3];
            objects[p].perihelionVLat = xperi[4];
            objects[p].aphelionLon    = xaphe[0];	// (peri + 180 for Sun and Moon)
            objects[p].aphelionLat    = xaphe[1];	// (-peri      for Sun and Moon)
            objects[p].aphelionVLon   = xaphe[3];	// (peri + 0   for Sun and Moon)
            objects[p].aphelionVLat   = xaphe[4];	// (-peri      for Sun and Moon)

            /*{   char	s[40];
            swe_get_planet_name(p, s);
            NSLog(@"%10s asc=%f %f %f %f %f %f", s, xnasc[0], xnasc[1], xnasc[2], xnasc[3], xnasc[4], xnasc[5]);
            NSLog(@"%10s dsc=%f %f %f %f %f %f", s, xndsc[0], xndsc[1], xndsc[2], xndsc[3], xndsc[4], xndsc[5]);
            NSLog(@"%10s peri=%f %f %f %f %f %f", s, xperi[0], xperi[1], xperi[2], xperi[3], xperi[4], xperi[5]);
            NSLog(@"%10s aphe=%f %f %f %f %f %f", s, xaphe[0], xaphe[1], xaphe[2], xaphe[3], xaphe[4], xaphe[5]);
            }*/
        }
        else
        {   objects[p].nodeLon = objects[p].nodeLat = objects[p].nodeVLon = objects[p].nodeVLat = 0.0;
            objects[p].perihelionLon  = objects[p].perihelionLat  = 0.0;
            objects[p].perihelionVLon = objects[p].perihelionVLat = 0.0;
            objects[p].aphelionLon    = objects[p].aphelionLat    = 0.0;
            objects[p].aphelionVLon   = objects[p].aphelionVLat   = 0.0;
        }

#if 0
        /* equatorial node + perigee */
        if (ephFlags & EPHFALG_EQU)
        {   double	xnasc[6], xndsc[6], xperi[6], xaphe[6];

            if (p >= SE_MEAN_NODE && p <= SE_OSCU_APOG)	// skip Moon node/apogee
                continue;

            /* equatorial (SEFLG_EQUATORIAL) */
            if ( (iflgret = swe_nod_aps_ut(tjd, p, iflag|SEFLG_EQUATORIAL,
                                           nodeBit, xnasc, xndsc, xperi, xaphe, serr)) < 0 )
                NSLog(@"error: %s (Nodes equatorial).\n", serr);
            /*else if (iflgret != (iflag|SEFLG_EQUATORIAL))
            NSLog(@"warning: iflgret (%f) != iflag (%f) (Nodes equatorial).\n",
                  iflgret, iflag|SEFLG_EQUATORIAL);*/
            //objects[p].nodeRec     = xnasc[0];	// rectascension
            objects[p].nodeDec     = xnasc[1];		// declination
            //objects[p].nodeDist    = xnasc[2];	// distance in AU
            //objects[p].nodeVRec    = xnasc[3];	// speed rectascension
            objects[p].nodeVDec    = xnasc[4];		// speed declination
            //objects[p].nodeVDist   = xnasc[5];	// speed distance
            //objects[p].dscNodeRec  = xndsc[0];	
            objects[p].dscNodeDec  = xndsc[1];
            //objects[p].dscNodeDist = xndsc[2];
            //objects[p].dscNodeVRec = xndsc[3];
            objects[p].dscNodeVDec = xndsc[4];
            //objects[p].dscNodeVDist = xndsc[5];

            objects[p].perigeeLon   = xperi[0];
            objects[p].perigeeLat   = xperi[1];
            objects[p].perigeeVLon  = xperi[3];
            objects[p].perigeeVLat  = xperi[4];
            objects[p].apogeeLon    = xaphe[0];
            objects[p].apogeeLat    = xaphe[1];
            objects[p].apogeeVLon   = xaphe[3];
            objects[p].apogeeVLat   = xaphe[4];
        }
#endif

        /* get the name of the planet p, debugging */
        /*{   char	snam[40];

            swe_get_planet_name(p, snam);
            printf("%10s\t%11.7f\t%10.7f\t%10.7f\t%10.7f\n", snam, x2[0], x2[1], x2[2], x2[3]);
        }*/
    }

    /* sun node */
    if (ephFlags & EPHFLAG_NOD)
    {   int	year, m, d;
        double	hour;

        swe_revjul (tjd, 1, &year, &m, &d, &hour);
        objects[SE_SUN].nodeLon = sunNode(year);
        objects[SE_SUN].nodeLat = 0.0;
    }
    else
        objects[SE_SUN].nodeLon = objects[SE_SUN].nodeLat = 0.0;

    /* add ac, mc */
    if (ephFlags & EPHFLAG_LOC && lon >= -180.0 && lat >= -90.0)
    {   double	st = vhfSideralTime(tjd, lon);

        objects[O_AC].lon  = vhfAC(st, lat);
        objects[O_AC].vLon = 1.0;
        objects[O_AC].vLat = 0.0;
        objects[O_MC].lon  = vhfMC(st);
        objects[O_MC].vLon = 1.0;
        objects[O_MC].vLat = 0.0;
    }
    else
    {   objects[O_AC].lon = objects[O_AC].vLon = objects[O_AC].vLat = 0.0;
        objects[O_MC].lon = objects[O_MC].vLon = objects[O_MC].vLat = 0.0;
    }

    return objects;
}

- (NSMutableDictionary*)objectDictAtUTC:(NSCalendarDate*)utc lat:(float)lat lon:(float)lon
{   NSCalendarDate      *st = sideralTime(utc, lon);
    NSMutableDictionary	*degreeDict = [self objectDictAtUTC:utc];

    [degreeDict setObject:[NSNumber numberWithDouble:ac(st, lat)] forKey:@"AC"];
    [degreeDict setObject:[NSNumber numberWithDouble:mc(st)]      forKey:@"MC"];
    return degreeDict;
}

- (NSMutableDictionary*)objectDictAtUTC:(NSCalendarDate*)utc lat:(double)lat lon:(double)lon
                                   elev:(double)alt topocentric:(BOOL)topocentric;
{   NSMutableDictionary *degreeDict = [NSMutableDictionary dictionary];
    int                 p;
    EphObject           *objs;
    //NSCalendarDate    *st = sideralTime(utc, lon);

    objs = [self objectsAtUTC:utc lat:lat lon:lon elev:alt topocentric:topocentric];

    for (p = SE_SUN; p <= SE_VESTA; p++)
        [degreeDict setObject:[NSNumber numberWithDouble:objs[p].lon]
                       forKey:[objectNames objectAtIndex:p]];
    [degreeDict setObject:[NSNumber numberWithDouble:objs[O_AC].lon] forKey:@"AC"];
    [degreeDict setObject:[NSNumber numberWithDouble:objs[O_MC].lon] forKey:@"MC"];
    //[degreeDict setObject:[NSNumber numberWithDouble:ac(st, lat)] forKey:@"AC"];
    //[degreeDict setObject:[NSNumber numberWithDouble:mc(st)] forKey:@"MC"];

    calcJD   = [self julianDayForDate:utc bc:NO];
    calcLat  = lat;
    calcLon  = lon;
    calcElev = alt;
    calcTopo = topocentric;

    return degreeDict;
}

/* is dictionary already calculated ?
 * created: 2006-08-18
 */
- (BOOL)calculatedAtDate:(NSCalendarDate*)date tolerance:(int)tol
                     lat:(float)lat lon:(float)lon elev:(float)elev topocentric:(BOOL)topo
{
    if ( ( (topo || calcTopo) && calcTopo == topo
           && Diff(elev, calcElev) < TOLERANCE
           && Diff(lat, calcLat) < TOLERANCE && Diff(lon, calcLon) < TOLERANCE )
         && Diff([self julianDayForDate:date bc:NO], calcJD) <= (double)tol/(24.0*60.0) )
        return YES;
    return NO;
}

- (NSMutableDictionary*)objectDictAtUTC:(NSCalendarDate*)utc
{   NSMutableDictionary		*degreeDict = [NSMutableDictionary dictionary];
    int				p;
    EphObject			*objs = [self objectsAtUTC:utc];

    for (p = SE_SUN; p <= SE_VESTA; p++)
        [degreeDict setObject:[NSNumber numberWithDouble:objs[p].lon]
                       forKey:[objectNames objectAtIndex:p]];

    calcJD   = [self julianDayForDate:utc bc:NO];
    calcLat  = calcLon = 0.0;
    calcElev = 0.0;
    calcTopo = NO;

    return degreeDict;
}

/* get current objects
 */
- (NSMutableDictionary*)currentObjectDict
{   NSMutableDictionary		*degreeDict = [NSMutableDictionary dictionary];
    int				p;

    for (p = SE_SUN; p <= SE_VESTA; p++)
        [degreeDict setObject:[NSNumber numberWithDouble:objects[p].lon]
                       forKey:[objectNames objectAtIndex:p]];
    return degreeDict;
}
/* get speed from current objects
 */
- (NSMutableDictionary*)currentSpeedLonDict
{   NSMutableDictionary		*speedDict = [NSMutableDictionary dictionary];
    int				p;

    for (p = SE_SUN; p <= SE_VESTA; p++)
        [speedDict setObject:[NSNumber numberWithDouble:objects[p].vLon]
                      forKey:[objectNames objectAtIndex:p]];
    return speedDict;
}
/* get nodes from current objects
 */
- (NSMutableDictionary*)currentNodeDict
{   NSMutableDictionary		*degreeDict = [NSMutableDictionary dictionary];
    int				p;

    for (p = SE_SUN; p <= SE_VESTA; p++)
        [degreeDict setObject:[NSNumber numberWithDouble:objects[p].nodeLon]
                       forKey:[objectNames objectAtIndex:p]];
    return degreeDict;
}
/* get declination from current objects
 */
- (NSMutableDictionary*)currentDeclinationDict
{   NSMutableDictionary		*degreeDict = [NSMutableDictionary dictionary];
    int				p;

    for (p = SE_SUN; p <= SE_VESTA; p++)
        [degreeDict setObject:[NSNumber numberWithDouble:objects[p].dec]
                       forKey:[objectNames objectAtIndex:p]];
    return degreeDict;
}
/* get declination speed from current objects
 */
- (NSMutableDictionary*)currentSpeedDecDict
{   NSMutableDictionary		*speedDict = [NSMutableDictionary dictionary];
    int				p;

    for (p = SE_SUN; p <= SE_VESTA; p++)
        [speedDict setObject:[NSNumber numberWithDouble:objects[p].vDec]
                      forKey:[objectNames objectAtIndex:p]];
    return speedDict;
}

/* get houses
 * we use the Topocentric house system
 * FIXME: houseCnt is not used
 */
- (double*)houseAtUTC:(NSCalendarDate*)utc latitude:(float)lat longitude:(float)lon houseCount:(int)houseCnt
{   double		tjd, h, ascmc[10];
    static double	cusps[13];

    /* we have day, month and year and convert to Julian day number */
    h = (double)[utc hourOfDay] + ((double)[utc minuteOfHour]*60.0 + (double)[utc secondOfMinute]) / 3600.0;
    tjd = swe_julday([utc yearOfCommonEra], [utc monthOfYear], [utc dayOfMonth], h, SE_GREG_CAL);

    swe_houses(tjd, lat, lon, 'T', cusps, ascmc);

    return cusps;
}

@end
