/* AspectLines.m
 * This calculates the traditional aspect lines
 *
 * Copyright (C) 2006 by vhf interservice GmbH
 * Author:   Georg Fleischmann
 *
 * created:  2006-04-11
 * modified: 2006-05-01
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 *
 * vhf interservice GmbH, Im Marxle 3, 72119 Ammerbuch, Germany
 * eMail: info@vhf.de
 * http://www.vhf.de
 */

#include <AppKit/AppKit.h>
#include <VHFShared/VHFArrayAdditions.h>
#include <VHFShared/VHFDictionaryAdditions.h>
#include "../Cenon/Graphics.h"
#include "astroCommon.h"
#include "AspectLines.h"

static AspectLines	*sharedInstance = nil;

/* Private methods
 */
@interface AspectLines(PrivateMethods)
@end

@implementation AspectLines

/* create new instance of GeneralController
 * created: 2005-07-27
 */
+ (id)sharedInstance
{
    if (!sharedInstance)
        sharedInstance = [[AspectLines alloc] init];
    return sharedInstance;
}

- (id)init
{
    [super init];

    [self setDividersFromString:@"1-6"];
    saturation = 1.0;

    showMirrors = YES;

    showStrengthOfOrb = YES;
    orb = 0.0275;		// 2.75% orb of divided segment (10 deg for 360 degree)
    maxOrbWidth = 2.0;

    showColoredDividers = YES;
    colors = [[NSArray arrayWithObjects:
        [NSColor blackColor],			//  1
        [NSColor greenColor],			//  2
        [NSColor redColor],			//  3
        [NSColor blueColor],			//  4
        [NSColor brownColor],			//  5
        [NSColor orangeColor],			//  6  (related to 3)
        [NSColor grayColor],			//  7
        [NSColor cyanColor],			//  8  (related to 4)
        //[NSColor colorWithCalibratedRed:0.0 green:0.0 blue:0.5 alpha:1.0],	//  8 = 0.5 * 4
        [NSColor orangeColor],			//  9
        [NSColor yellowColor],			// 10
        [NSColor purpleColor],			// 11
        [NSColor magentaColor], nil] retain];	// 12

    return self;
}



/* set dividers from string
 * divStr = "1 2-6 9"
 */
- (void)setDividersFromString:(NSString*)divStr
{   NSScanner		*scanner  = [NSScanner scannerWithString:divStr];
    NSMutableArray	*dividers = [NSMutableArray array];
    NSString		*string;
    NSCharacterSet	*whitespaceCharacterSet = [NSCharacterSet whitespaceCharacterSet];
    NSRange		range;
    int			i;

    while (![scanner isAtEnd])
    {
        if (![scanner scanUpToCharactersFromSet:whitespaceCharacterSet intoString:&string])
            break;

        /* '-' -> range of dividers */
        range = [string rangeOfString:@"-"];
        if (range.length)
        {   int	d, dMin, dMax;

            dMin = [[string substringToIndex:range.location] intValue];
            dMax = Min([[string substringFromIndex:range.location+1] intValue], 360);
            for (d=dMin; d<=dMax; d++)
            {   NSNumber	*divNum = [NSNumber numberWithInt:d];
                if ( ! [dividers containsObject:divNum] )
                    [dividers addObject:divNum];
            }
        }
        /* '#' -> divider */
        else
        {   NSNumber	*divNum = [NSNumber numberWithInt:[string intValue]];
            if ( ! [dividers containsObject:divNum] )
                [dividers addObject:divNum];
        }
    }

    dividerCnt = [dividers count];
    if (dividerCnt >= MAX_DIVIDER)
    {   NSLog(@"Number of dividers exceeds fixed limits %d > %d", dividerCnt, MAX_DIVIDER);
        dividerCnt = MAX_DIVIDER;
    }

    [dividers sortUsingSelector:@selector(compare:)];

    /* store dividers (1 2 3 4 5 6) */
    for (i=0; i<dividerCnt; i++)
        divider[i] = [dividers intAtIndex:i];
}

- (void)setDisplayMirrors:(BOOL)mirrorFlag
{
    showMirrors = mirrorFlag;
}

- (void)setStrengthOfOrb:(BOOL)orbFlag
{
    showStrengthOfOrb = orbFlag;
}

- (void)setColoredDividers:(BOOL)flag
{
    showColoredDividers = flag;
}

/* calculate basic size from radius using the biggest divider
 */
- (void)setRadius:(float)newRadius
{
    radius = newRadius;
}

- (void)setColorSaturation:(float)s
{
    saturation = s;
}

/* the objects we work with - we store a pointer only !
 */
- (void)setObjects:(double*)newObjs  cnt:(int)oCnt
{
    objects = newObjs;
    objectCnt = oCnt;
}

/* the relative orb
 */
- (void)setOrb:(float)newOrb
{
    if (orb >= 0 && orb <= 1.0)
        orb = newOrb;
    else
        NSLog(@"AspectLines: Orb out of range [0, 1.0], not changed!");
}


/* remainder = num % denom
 */
static double modulo(double num, double denom)
{
    return (num - floor(num / denom) * denom);
}

/* Aspect Lines
 *
 * This thingy works as follows:
 * - conjunction = ball
 * - other aspects are displayed as a connecting line
 * - thickness of lines may tell something about the orb
 * - the color may tell something about the aspect
 *
 * created:  2006-04-11
 * modified: 2006-04-15
 */
#define MIN_LINEWIDTH	0.2
- (VGroup*)calculateAtCenter:(NSPoint)center
{   VGroup	*group = [VGroup group];
    int		i, n, m, a;
    float	rad = radius * 0.95;	// inner radius for lines

    if (!dividerCnt)
        return nil;

    /* draw circle around aspect lines */
    {   VArc	*arc;

        arc = [VArc arcWithCenter:center radius:radius filled:NO];
        //[arc setFillColor:[NSColor colorWithDeviceWhite:0.95 alpha:1.0]];
        [arc setWidth:MIN_LINEWIDTH];
        [group addObject:arc];
    }

    /* draw scale lines at object positions */
    for (n=0; n<objectCnt; n++)
    {   VLine	*line;
        float	deg = objects[n];
        NSPoint	p0, p1;

        p0 = NSMakePoint(center.x + radius, center.y);
        p0 = vhfPointRotatedAroundCenter(p0, deg, center);
        p1 = NSMakePoint(center.x + rad, center.y);
        p1 = vhfPointRotatedAroundCenter(p1, deg, center);
        line = [VLine lineWithPoints:p0 :p1];
        [line setWidth:MIN_LINEWIDTH];
        [group addObject:line];
    }

    /* loop of dividers */
    //for (i=0; i<dividerCnt; i++)
    for (i=dividerCnt-1; i>=0; i--)			// with this larger dividers cover smaller dividers
    {   int	div = divider[i];			// divider of this concentric layer
        float	waveLen = 360.0 / div;			// wave length for this divider
        //float	orbV = waveLen * orb;			// orb in degree

        /* loop of first object */
        for (n=0; n<objectCnt; n++)			// only objects are allowed masters !
        {   float	deg1 = objects[n];		// master degree / reference

            /* loop of second object */
            for (m=n+1; m<objectCnt; m++)
            {   float	deg2 = objects[m], ang = 0.0, gap;

                for (a=0; a < ((showMirrors) ? 2 : 1); a++)
                {
                    switch (a)
                    {
                        case 0:	// angle
                            ang = angle(deg2, deg1);
                            break;
                        case 1:	// mirror
                            if (div > 1)	// more would clutter everything with lines
                                break;
                            ang = mirror(deg2, deg1);
                    }
                    gap = modulo(ang, waveLen);		// overlap = angle % waveLen
                    gap = Min(gap, waveLen - gap);	// smallest overlap to this or next node

                    if ( gap / waveLen < orb )
                    {   VLine		*line;
                        NSPoint		p, pts[2];
                        float		exactness = 1.0 - ((gap/waveLen) / orb);	// 1.0 = exact

                        /* create a line between corrected degrees of each set, close the circle */
                        p = NSMakePoint(center.x + rad, center.y);
                        pts[0] = vhfPointRotatedAroundCenter(p, deg1, center);
                        pts[1] = vhfPointRotatedAroundCenter(p, deg2, center);
                        if (div > 1 || a == 1)
                        {   line = [VLine lineWithPoints:pts[0] :pts[1]];
                            if (showStrengthOfOrb)		// orb strength
                                [line setWidth:maxOrbWidth * exactness];
                            else
                                [line setWidth:0.3];
                            if (showColoredDividers)		// aspect color
                            {   NSColor	*color = [colors objectAtIndex:Min(div-1, [colors count]-1)];
                                [line setColor:color];
                            }
                            [group addObject:line];
                        }
                        else		// conjunction
                        {
                            // FIXME
                        }
                    }
                } // angle / mirror
            } // m loop
        } // n loop
    } // divider loop

    return group;
}


- (void)dealloc
{
    if (self == sharedInstance)
        return;
    [colors release];
    [super dealloc];
}

@end
