/* EphemerisSearch.m
 * search Ephemeris data
 *
 * Copyright (C) 2004-2009 by vhf interservice GmbH
 * Author:   Georg Fleischmann
 *
 * created:  2004-04-12
 * modified: 2009-02-22
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the vhf Public License as
 * published by vhf interservice GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the vhf Public License for more details.
 *
 * You should have received a copy of the vhf Public License along
 * with this program; see the file LICENSE. If not, write to vhf.
 *
 * vhf interservice GmbH, Im Marxle 3, 72119 Altingen, Germany
 * eMail: info@vhf.de
 * http://www.vhf.de
 */

#include <Foundation/Foundation.h>
#include <VHFShared/types.h>
#include <VHFShared/VHFDictionaryAdditions.h>
#include "EphemerisSearch.h"
#include "astroCommon.h"

@interface EphemerisSearch(PrivateMethods)
@end

@implementation EphemerisSearch

#define DEBUG_SEARCH	0	// display debug info for aspect search

/*static float objectDegPerDay[O_ARRAYSIZE] =
 { 0.9856, 13.1868,  4.0955,  1.6021,   0.5240,		// sun - mars
   0.08367, 0.03346, 0.01173, 0.009406, 0.00397,	// jupiter - pluto
   0.05305, 0.05305, 0.11137, 0.11137,			// node, apogee
   0.02011, 0, 0, 0, 0, 0				// chiron
 };*/

/* seconds/degree = tCycle [secs] / 360 deg
 * seconds/degree = (24*3600) / degPerDay
 */
static unsigned secsPerDegree[O_ARRAYSIZE] =		// average seconds per degree of arc
 {    87660,     6552,   21120,   53929,    164885,	// sun - mars
    1032628,  2582188, 7365729, 9185626, 217632242,	// jupiter - pluto
   16286522, 16286522,  775757,  775757,		// node, apogee
    4296370, 0, 0, 0, 0, 0				// chiron
 };
//#define SecsPerDegree	[NSArray arrayWithObjects:@"87660", @"6552", @"21120", @"53929", @"164885", @"1032628", @"2582188", @"7365729", @"9185626", @"217632242", @"16286522", @"16286522", @"775757", @"775757", @"87660", @"4296370", @"Pholus", @"Ceres", @"Pallas", @"Juno", @"Vesta", nil]

+ newWithEphemeris:(Ephemeris*)eph
{
    return [[self alloc] initWithEphemeris:eph];
}

- (id)initWithEphemeris:(Ephemeris*)eph
{
    [super init];
    ephemeris = [eph retain];

    topoLon = topoLat = 0.0;
    topoElev = 0;

    return self;
}
- (void)setProgressTarget:(id)target action:(SEL)action
{
    progressTarget = target;
    progressAction = action;
}

- (void)setTopocentricSearch:(BOOL)topoFlag lat:(float)lat lon:(float)lon elev:(int)elev
{
    topocentricSearch = topoFlag;
    if (topocentricSearch)
    {
        topoLat  = lat;
        topoLon  = lon;
        topoElev = elev;
    }
}


/* cycle length in minutes
 * created: 2006-01-10
 */
- (unsigned)minutesCycleForAspect:(EphemerisObjects)obj1 :(EphemerisObjects)obj2
                           aspect:(SearchAspectType)aspType
{   unsigned	min = 0;

    switch (aspType)
    {
        case S_ASPECT_LON:
            min = secsPerDegree[obj1] / 60 * 360;
            //min = 360.0 / objectDegPerDay[obj1] * 24 * 60;
            break;
        // FIXME: all other aspects
        default:
            NSLog(@"[EphemerisSearch minutesPerCycleForAspect]: not implemented yet!");
            break;
    }
    return min;
}

/* retrograde length and interval (direct) in minutes
 *  Mercury     every  4 months for  2 weeks (Mercury mostly)
 *  Venus       every 19 months for  5 weeks (Venus + Sun)
 *  Mars        every 25 months for  8 weeks (Sun + Mars)
 *  Jupiter     every 12 months for 16 weeks (Sun mostly)
 *  Saturn      every 12 months for 16 weeks (Sun mostly)
 *  ...
 *  Moon Apogee every  1 month  for  2 weeks (Moon, month = 28 days here)
 *  Moon Node   every  2 weeks  for 2-5 days (Moon, direct times for Node!)
 *
 * created: 2006-01-19
 */
static unsigned hoursRetro[O_ARRAYSIZE] =	// retrograde for n hours
 {       0,       0,  2*7*24,  5*7*24,  8*7*24,			// sun - mars
   16*7*24, 16*7*24, 16*7*24, 16*7*24, 16*7*24,			// jupiter - pluto
         0,    2*24,       0,  2*7*24,				// node, true, apogee, true
   16*7*24, 16*7*24, 16*7*24, 16*7*24, 16*7*24, 16*7*24 	// chiron
 };
static unsigned hoursDirect[O_ARRAYSIZE] =	// intervals between retrograde in hours
 {        0,        0,  4*30*24, 19*30*24, 25*30*24,		// sun - mars
   12*30*24, 12*30*24, 12*30*24, 12*30*24, 12*30*24,		// jupiter - pluto
          0,   2*7*24,        0,   2*7*24,			// node, true, apogee, true
   12*30*24, 12*30*24, 12*30*24, 12*30*24, 12*30*24, 12*30*24 	// chiron
 };
- (unsigned)minutesRetroForAspect:(EphemerisObjects)obj1 :(EphemerisObjects)obj2
                           aspect:(SearchAspectType)aspType
                           direct:(unsigned*)direct
{   unsigned	min = 0;

    switch (aspType)
    {
        case S_ASPECT_LON:
            min     = hoursRetro[obj1]  * 60;
            *direct = hoursDirect[obj1] * 60;
            break;
        // FIXME: all other aspects
        default:
            NSLog(@"[EphemerisSearch minutesRetroForAspect]: not implemented yet!");
            break;
    }
    return min;
}


/* return value of aspect
 * created:  2006-01-11 (from trader module)
 * modified: 2006-02-03
 */
- (AstroAspectValue)valueOfAspect:(int)obj1 :(int)obj2 aspect:(SearchAspectType)asp
                           atDate:(NSCalendarDate*)date
{   //static EphObject	ephBuffer[2];
    double              v1, v2;
    EphemerisFlags      ephFlags = EPHFLAG_LON;	// what to calculate
    EphemerisObjects    ephObjIx[2];		// objects to calculate
    int                 ephObjCnt = 0;		// number of objects in array
    EphObject           *eph; //eph[O_ARRAYSIZE];
    AstroAspectValue    value = {0.0, 0.0};

    if (obj1 != EPH_NONE)
        ephObjIx[ephObjCnt++] = obj1;
    if (obj2 != EPH_NONE)
        ephObjIx[ephObjCnt++] = obj2;
    switch (asp)
    {
        default: break;
        case S_ASPECT_DEC:		// declination aspects
        case S_ASPECT_PARALLEL:
        case S_ASPECT_DEC_ANGLE:
            ephFlags = EPHFLAG_DEC;
    }

    if (1)
    {   //EphObject	*ephObject;

        eph = [ephemeris objectsAtJD:[ephemeris julianDayForDate:date bc:NO] flags:ephFlags
                             objects:ephObjIx cnt:ephObjCnt
                           longitude:topoLon latitude:topoLat elevation:topoElev
                         topocentric:topocentricSearch];
        //for (i=0; i<O_ARRAYSIZE; i++)
        //    eph[i] = ephObject[i];
    }

    switch (asp)
    {
        case S_ASPECT_LON:		// object on ecliptic
            value.val = eph[obj1].lon;
            value.v   = eph[obj1].vLon;
            break;
        case S_ASPECT_DEC:		// object declination (x 3 to expand resolution)
            value.val = eph[obj1].dec;
            value.v   = eph[obj1].vDec;
            break;
        case S_ASPECT_LOCAL:		// object in houses
            NSLog(@"degreeOfAspect: local aspects not yet implemented!");
            //value.val = topo->deg[obj1];
            break;

        case S_ASPECT_ANGLE:		// object angles
            value.val = angle(eph[obj1].lon, eph[obj2].lon);
            value.v   = eph[obj1].vLon + eph[obj2].vLon;	// FIXME
            break;
        case S_ASPECT_MIRROR:		// object mirrors (obj1 - (360 - obj2))
            value.val = mirror(eph[obj1].lon, eph[obj2].lon);
            break;
        case S_ASPECT_PARALLEL:		// object parallels (declination mirror)
            NSLog(@"degreeOfAspect: parallels not yet implemented!");
            v1 = eph[obj1].dec; if (v1 < 0.0) v1 = 360.0+v1;		// FIXME
            v2 = eph[obj2].dec; if (v2 < 0.0) v2 = 360.0+v2;
            value.val = mirror(v1, v2);
            //v1 = eph[obj1].vDec; if (v1 < 0.0) v1 = 360.0+v1;		// FIXME
            //v2 = eph[obj2].vDec; if (v2 < 0.0) v2 = 360.0+v2;
            //value.v = mirror(v1, v2);
            break;
        case S_ASPECT_DEC_ANGLE:		// declination angle
            NSLog(@"degreeOfAspect: declination angle not yet implemented!");
            v1 = eph[obj1].dec; if (v1 < 0.0) v1 = 360.0+v1;		// FIXME
            v2 = eph[obj2].dec; if (v2 < 0.0) v2 = 360.0+v2;
            value.val = angle(v1, v2);
            //v1 = eph[obj1].vDec; if (v1 < 0.0) v1 = 360.0+v1;		// FIXME
            //v2 = eph[obj2].vDec; if (v2 < 0.0) v2 = 360.0+v2;
            //value.v = angle(v1, v2);
            break;
        case S_ASPECT_LOCAL_ANGLE:	// local angle
        case S_ASPECT_LOCAL_MIRROR:	// local mirror
            NSLog(@"degreeOfAspect: local aspects not yet implemented!");	// FIXME
            //[localTopo setDate:yymmdd time:hhmm latitude:latitude longitude:longitude];
            //v1 = [localTopo topocentricDegree:eph[obj1].lon];
            //v2 = [localTopo topocentricDegree:eph[obj2].lon];
            /*v1 = topo->deg[obj1];
            v2 = topo->deg[obj2];
            value.val = (aspectType == ASPECT_LOCAL_ANGLE) ? angle(v1, v2) : mirror(v1, v2);*/
            break;

        default:
            NSLog(@"degreeOfAspect: aspect not yet implemented!");	// FIXME
            break;
    }
    return value;
}

/* binary search for a single aspect
 *
 * results =
 * (
 *    date1 = deg;
 *    date2 = deg;	// 2 or more for several passages, mark retrograde passages ?
 * )
 *
 * created:  2006-01-10
 * modified: 2006-02-26
 */
- (NSArray*)searchFromDate:(NSCalendarDate*)dateFr toDate:(NSCalendarDate*)dateTo
                   object1:(EphemerisObjects)obj1 object2:(EphemerisObjects)obj2
                    aspect:(SearchAspectType)aspType aspectValue:(double)aspVal
{   NSMutableArray      *results = [NSMutableArray array];
    //int               years = [dateTo yearOfCommonEra] - [dateFr yearOfCommonEra] + 1;
    NSCalendarDate      *date;
    int                 i, interval, minCycle, minRetro, loop = 0, loopCnt = 0;
    int                 secCnt = 0;
    unsigned int        minDirect;
    EphObject           *objects;
    BOOL                retroAspect = NO, retro = NO;	// wheather aspect was retrograde in test sections
    NSMutableArray      *sectionDates;
    EphemerisFlags      ephFlags = 0;	// what to calculate
    EphemerisObjects    ephObjIx[2];	// objects to calculate
    int                 ephObjCnt = 0;	// number of objects in array
    double              tjd;

    if (obj1 != EPH_NONE)
        ephObjIx[ephObjCnt++] = obj1;
    if (obj2 != EPH_NONE)
        ephObjIx[ephObjCnt++] = obj2;
    switch (aspType)
    {
        default:
        case S_ASPECT_LON:		// longitude aspects
            ephFlags = EPHFLAG_LON;
            break;
        case S_ASPECT_DEC:		// declination aspects
        case S_ASPECT_PARALLEL:
        case S_ASPECT_DEC_ANGLE:
            ephFlags = EPHFLAG_DEC;
    }

    /* binary search through range
     * 1. divide search range into sections of roughly one cycle for each the aspect.
     *    further divide each section to surely cover all retrogrades of the aspects.
     *    If we have found a retro section, we make a larger gap to be sure we are behind.
     * 2. a. determine turn points around target (between previous and next retro).
     *    b. build <= 3 sections from <= 2 turn points around target.
     * 3. perform binary search inside each linear section.
     * Examples:
     *    - Mercury is retrograde every 4 months (120 deg) for about 2-4 weeks, when ~16deg near Sun.
     *      -> 25 test searches (every 2 weeks) to get all retros per year (8 retros)
     *         + 2 x 19 searches to get turn points before and after target (2 turn points), 1min exact
     *         + 3 x 19 searches to get all targets inside target sections (3 section), 1min exact
     *         = < 120 searches/year to get 1 minute resolution
     *    - Venus is retrograde about once a year for 4-6 weeks
     *    - Mars every 2 years for ~10 weeks
     *    - Uranus each year for ~5 months
     */

    /* 1. divide search range into smaller sections
     */
    /*    we use smallest section (cycle, retro) to split from start to end of range */
    minCycle = [self minutesCycleForAspect:obj1 :obj2 aspect:aspType];
    minCycle -= minCycle / 10;	// 10% tolerance
    minRetro = [self minutesRetroForAspect:obj1 :obj2 aspect:aspType direct:&minDirect];
    interval = (minRetro > 0) ? Min(minCycle, minRetro-minRetro/5) : minCycle;
    sectionDates = [NSMutableArray array];
    for ( date = dateFr;					// interval steps through range
          [date compare:dateTo] != NSOrderedDescending; )	// date <= dateTo
    {   int	step, min = 0;

        if (secCnt)
            min = [date timeIntervalSinceDate:[sectionDates objectAtIndex:secCnt-1]] / 60.0;
        //objects = [ephemeris objectsAtUTC:date
        //                              lat:topoLat lon:topoLon elev:topoElev
        //                      topocentric:topocentricSearch];
        tjd = [ephemeris julianDayForDate:date bc:NO];
        objects = [ephemeris objectsAtJD:tjd flags:ephFlags
                                 objects:ephObjIx cnt:ephObjCnt
                               longitude:topoLon latitude:topoLat elevation:topoElev
                             topocentric:topocentricSearch];
        /* add section date only, if retro changes or delta >= 360deg */
        if ( !secCnt || min >= minCycle ||			// delta >= full cycle
             retro != ((objects[obj1].vLon < 0.0) ? YES : NO) )	// retro changes
        {
            [sectionDates addObject:date];
            secCnt ++;
        }
        // FIXME: switch (aspType) -> add other aspects
        if (objects[obj1].vLon < 0.0)		// retrograde
        {   step = minDirect - minDirect/5;	// step with some safety
            retroAspect = retro = YES;
        }
        else
        {   step = interval;
            retro = NO;
        }
        date = [date dateByAddingYears:0 months:0 days:0 hours:0 minutes:step seconds:0];
    }
    /* add end of range */
    if ( secCnt && ![[sectionDates objectAtIndex:secCnt-1] isEqual:dateTo] )
        [sectionDates addObject:dateTo];
    secCnt = [sectionDates count];

    /* 2. find turn points (binary search) and create linear sections
     */
    if (retroAspect)	// was retorgrade during test
    {   NSMutableArray	*turnDates = [NSMutableArray array];

#if DEBUG_SEARCH
        NSLog(@"Section Dates = %@", sectionDates);
#endif
        [progressTarget performSelector:@selector(setTitle:) withObject:@"Pass 1"];
        loop = 0; loopCnt = secCnt*20*2;	// 1st 50% of progress bar
        [turnDates addObject:dateFr];
        for (i=0; i<secCnt-1; i++)	// for each section
        {   NSCalendarDate	*secDateFr, *secDateTo;
            int			j;
            AstroAspectValue	minV, v, maxV;
            NSCalendarDate	*minDate, *maxDate, *date;

            secDateFr = [sectionDates objectAtIndex:i];		// start date of section
            secDateTo = [sectionDates objectAtIndex:i+1];	// end   date of section

            /* binary search for turn points (v == 0) */
            minDate = secDateFr;
            maxDate = secDateTo;
            minV = [self valueOfAspect:obj1 :obj2 aspect:aspType atDate:minDate];
            maxV = [self valueOfAspect:obj1 :obj2 aspect:aspType atDate:maxDate];
            for (j=0; j<20; j++)	// binary search
            {   NSTimeInterval	secs = [maxDate timeIntervalSinceDate:minDate];

                date = [minDate addTimeInterval:secs/2];
                v = [self valueOfAspect:obj1 :obj2 aspect:aspType atDate:date];

                [progressTarget performSelector:progressAction
                                     withObject:[NSNumber numberWithFloat:(float)loop++/(float)loopCnt]];

                if (secs < 30 || v.v == 0.0)	// ready !
                    break;
                /* speed: < 0 = retrograde, 0, > 0 = direct
                 * minV < 0.0 && v < 0.0 -> minV = v
                 * minV > 0.0 && v > 0.0 -> minV = v
                 */
                if (minV.v * maxV.v > 0.0)	// one section is at a turn date
                {   NSLog(@"Search Turn Dates: Directions should be different: min = {%@  = %.4f}, v = {%@ = %.4f}, max = {%@ = %.4f}", minDate, minV.v, date, v.v, maxDate, maxV.v);
                    break;	// min/max directions shouldn't be the same
                }
                if ( (minV.v < 0.0 && v.v < 0.0) ||	// inside 1st half (min < v < target)
                     (minV.v > 0.0 && v.v > 0.0) )
                {   minV = v;
                    minDate = date;
                }
                else				// inside 2nd half
                {   maxV = v;
                    maxDate = date;
                }
            }
            loopCnt -= (20-j);	// adjust progress bar to shortcuts
            [turnDates addObject:date];
        }

        [turnDates addObject:dateTo];
#if DEBUG_SEARCH
        NSLog(@"Turn Dates = %@", turnDates);
#endif
        sectionDates = turnDates;
        secCnt = [turnDates count];
    }

    /* 3. binary search over linear section
     */
    if (retroAspect)
        [progressTarget performSelector:@selector(setTitle:) withObject:@"Pass 2"];
    loop    = (retroAspect) ? secCnt*20 : 0;
    loopCnt = secCnt*20 * ((retroAspect) ? 2 : 1);	// 2nd 50% of progress bar
    for (i=0; i<secCnt-1; i++)	// for each section
    {   NSCalendarDate	*secDateFr, *secDateTo;
        int			j;
        AstroAspectValue	minV, v, maxV;
        NSCalendarDate		*minDate, *maxDate, *date;
        BOOL			retro = NO;

        secDateFr = [sectionDates objectAtIndex:i];	// start date of section
        secDateTo = [sectionDates objectAtIndex:i+1];	// end   date of section

        /* actual binary search inside section */
        minDate = secDateFr;
        maxDate = secDateTo;
        minV = [self valueOfAspect:obj1 :obj2 aspect:aspType atDate:minDate];
        maxV = [self valueOfAspect:obj1 :obj2 aspect:aspType atDate:maxDate];

        for (j=0; j<20; j++)	// binary search
        {   NSTimeInterval	secs = [maxDate timeIntervalSinceDate:minDate];

            date = [minDate addTimeInterval:secs/2];
            v = [self valueOfAspect:obj1 :obj2 aspect:aspType atDate:date];
            if ( !j )
            {
                if (v.v < 0.0)	// retrograde
                    retro = YES;
                /* check, if section covers aspVal at all */
                if ( !vhfIsDegreeInsideRange(aspVal, minV.val, maxV.val, retro) )
                    /*skip = YES,*/ break;
            }

            [progressTarget performSelector:progressAction
                                 withObject:[NSNumber numberWithFloat:(float)loop++/(float)loopCnt]];

            if (secs < 30)	// ready !
                break;
            if ( vhfIsDegreeInsideRange(v.val, minV.val, aspVal, retro) )
            {   minV = v;
                minDate = date;
            }
            else					// inside 2nd half
            {   //maxV = v;
                maxDate = date;
            }
        }
        loopCnt -= (20-j);	// adjust progress bar to shortcuts
        if ( angle180(v.val, aspVal) < 1.0 )
        {
            /* FIXME: build average time and value to further improve accuracy */
            /* this has to be done between v (date) and minV/maxV (minDate/maxDate) */
            /*delta  = Diff(minV, v);		// delta interval
            delta1 = Diff(minV, aspVal);	// delta target
            dSec = [date timeIntervalSinceDate:minDate] * delta1 / delta;
            date0 = [date dateByAddingYears:0 months:0 days:0 hours:0 minutes:0 seconds:dSec];*/

            // FIXME: add mark for retrograde passages
            [results addObject:[NSArray arrayWithObjects:date, [NSNumber numberWithDouble:v.val], nil]];
        }
    }
    [progressTarget performSelector:progressAction
                         withObject:[NSNumber numberWithFloat:1.0]];

    [progressTarget performSelector:@selector(setTitle:) withObject:nil];

#if DEBUG_SEARCH
    NSLog(@"Search results = %@", results);
#endif
    return results;
}

/* return a dictionary with arrays containing dates and the events
 *
 * search features =
 * {
 *    entrySign = "1";	// search for sign ingress
 *    signCnt   = "24";	// default = 24 signs
 *    aspects   = "1";	// search for angles
 *    mirrors   = "1";	// search for mirrors
 *    dividers  = (1, 2, 3, ...);
 *    planets   = ("Sun", ...);
 *    radix     = { date = NSCalendarDate; lat = latitude; lon = longitude; };
 * }
 *
 * returned =
 * {
 *    planet = 
 *    {
 *      entrySign       = (date1 = sign#; date2 = sign#;);
 *      entryRadixHouse = (date1 = house#; date2 = house#;);
 *      aspects         = (date1 = (planet, devider); date2 = (planet, devider););
 *      aspectsRadix    = (date1 = (planet, devider); date2 = (planet, devider););
 *      mirrors         = (date1 = (planet, devider); date2 = (planet, devider););
 *      mirrorsRadix    = (date1 = (planet, devider); date2 = (planet, devider););
 *    };
 * }
 *
 * created:  2004-04-12
 * modified: 2006-01-09 (signCnt added)
 */
#define SIGNSIZE	15	// [degrees]
- (NSDictionary*)searchWithFeatures:(NSDictionary*)features
                           fromDate:(NSCalendarDate*)dateFrom toDate:(NSCalendarDate*)dateTo
                     searchInterval:(int)minutes
{   NSMutableDictionary	*dict = [NSMutableDictionary dictionary];
    NSCalendarDate	*date = dateFrom, *date0;
    NSArray		*planets = [features objectForKey:@"planets"];
    int			planetCnt = [planets count];
    int			p, p1, planetIx[planetCnt], i;
    /* sign entry */
    BOOL		featureEnterSigns = ([features objectForKey:@"entrySign"]) ? YES : NO;
    int			signSize = SIGNSIZE;	// default = 24 houses
    int			signNo[planetCnt];
    double		lastDeg[planetCnt];
    /* aspects (angles, mirrors) */
    BOOL		featureAspects = ([features objectForKey:@"aspects"])   ? YES : NO;
    BOOL		featureMirrors = ([features objectForKey:@"mirrors"])   ? YES : NO;
    NSArray		*dividers = [features objectForKey:@"dividers"];
    int			divCnt = [dividers count];
    double		target[planetCnt][planetCnt][divCnt],
                        lastAngle [planetCnt][planetCnt][divCnt], closestAngle [planetCnt][planetCnt][divCnt];
    double		lastMirror[planetCnt][planetCnt][divCnt], closestMirror[planetCnt][planetCnt][divCnt];

    if (!(featureEnterSigns || featureAspects || featureMirrors) || ![planets count])	// nothing to do
        return nil;

    if ([features intForKey:@"signCnt"])
        signSize = 360.0 / [features intForKey:@"signCnt"];

#if DEBUG_SEARCH
    NSLog(@"Search from: %@ to: %@\nfeatures = %@", dateFrom, dateTo, features);
#endif

    /* create dictionaries for planets */
    for (p=0; p<planetCnt; p++)
    {   NSMutableDictionary	*planetDict = [NSMutableDictionary dictionary];
        NSString		*planet = [planets objectAtIndex:p];

        [dict setObject:planetDict forKey:planet];
        planetIx[p] = [ephemeris indexForObjectName:planet];

        if (featureEnterSigns)
        {   [planetDict setObject:[NSMutableDictionary dictionary] forKey:@"entrySign"];
            signNo[p] = -1;
            lastDeg[p] = -1.0;
        }
        if (featureAspects || featureMirrors)
        {
            if (p < planetCnt-1)
                [planetDict setObject:[NSMutableDictionary dictionary] forKey:@"aspects"];
            dividers = [dividers sortedArrayUsingSelector:@selector(compare:)];
            for (p1=0; p1<[planets count]; p1++)
                for (i=0; i<divCnt; i++)
                {
                    target[p][p1][i] = 360.0 / [[dividers objectAtIndex:i] floatValue];
                    lastAngle [p][p1][i] = closestAngle [p][p1][i] = 1000.0;
                    lastMirror[p][p1][i] = closestMirror[p][p1][i] = 1000.0;
                }
        }
    }

    /* search through range */
    while ([date compare:dateTo] != NSOrderedDescending)	// date <= dateTo
    {   EphObject	*objects = [ephemeris objectsAtUTC:date];

        /* entry of sign */
        if (featureEnterSigns)
        {
            for (p=0; p<[planets count]; p++)
            {   double  deg = objects[planetIx[p]].lon;
                int	sign = deg / signSize;

                if (signNo[p] > -1 && sign != signNo[p])	// sign changed
                {   NSMutableDictionary	*entryDict;
                    double	v = deg, vPrev = lastDeg[p], delta, delta1;
                    int		dSec, targetSign;

//printf("%s: deg=%f (%f) sign = %d (%d)\n", [[date description] cString], deg, lastDeg[p], sign, signNo[p]);
                    /* get exact time by building average */
                    if (Diff(v, vPrev) >= 180.0)	// flip over 0
                    {
                        if (v < vPrev) vPrev -= 360.0;
                        else           v     -= 360.0;
                    }
                    targetSign = (v >= vPrev) ? sign : signNo[p];	// direct/retrograde
                    delta  = Diff(vPrev, v);				// delta interval
                    delta1 = Diff(vPrev, targetSign*signSize);		// delta target
                    dSec = minutes*60 * delta1 / delta  - (minutes*60.0);
                    date0 = [date dateByAddingYears:0 months:0 days:0 hours:0 minutes:0 seconds:dSec];

                    //printf("sign changed %s: signNo[%d]=%d sign=%d", [[date0 description] cString], p, signNo[p], sign);
                    entryDict = [[dict objectForKey:[planets objectAtIndex:p]] objectForKey:@"entrySign"];
                    targetSign = ((deg >= lastDeg[p]) ? 1.0 : -1.0) * targetSign;	// retrograde = negative
                    [entryDict setObject:[NSNumber numberWithInt:targetSign*signSize] forKey:date0];
                }
                signNo[p] = sign;
                lastDeg[p] = deg;
            }
        }

        /* aspects */
        if (featureAspects || featureMirrors)
        {
            for (p=0; p<planetCnt-1; p++)
            {   double  deg = objects[planetIx[p]].lon;

                for (p1=p+1; p1<planetCnt; p1++)
                {   double  deg1 = objects[planetIx[p1]].lon;

                    if (featureAspects)
                    {   double  a = angle(deg, deg1);

                        for (i=divCnt-1; i>=0; i--)	// use the biggest divider, if more than one divider fit
                        {   double	is  = a - floor(a / target[p][p1][i]) * target[p][p1][i];

                            // FIXME: we should allow approach from both directions: 0 and target !
                            //        closestAngle should be the diff to the node (0 or target)
                            if ( is <= lastAngle[p][p1][i] )
                                closestAngle[p][p1][i] = is;
                            else if (closestAngle[p][p1][i] < 3.0)
                            {   double			v, delta, delta1;
                                int			dSec;
                                NSMutableDictionary	*entryDict;
                                NSArray			*entry;

                                /* get exact date: date += deltaTime * delta1 / delta */
                                v = (is < closestAngle[p][p1][i]) ? is+target[p][p1][i] : is;
                                delta  = Diff(closestAngle[p][p1][i], target[p][p1][i]) + Diff(v, target[p][p1][i]);
                                delta1 = Diff(closestAngle[p][p1][i], target[p][p1][i]);
                                dSec = minutes*60 * delta1 / delta  - (minutes*60.0);
                                date0 = [date dateByAddingYears:0 months:0 days:0 hours:0 minutes:dSec/60 seconds:0];
                                //printf("%s %s  v=%f delta=%f delta1 = %f\n",
                                //[[date descriptionWithCalendarFormat:@"%Y%m%d %H:%M:%S"] cString],
                                //[[date0 descriptionWithCalendarFormat:@"%Y%m%d %H:%M:%S"] cString],
                                //v, delta, delta1);

                                v = floor((a+target[p][p1][i]/2.0)/target[p][p1][i]) * target[p][p1][i];
                                if (v >= 360.0) v = 0.0;
                                //printf("a=%f v=%f target[p][p1][i] = %f\n", a, v, target[p][p1][i]);
                                entry = [NSArray arrayWithObjects:[planets objectAtIndex:p1],
                                        [dividers objectAtIndex:i], [NSNumber numberWithDouble:v], nil];
                                entryDict = [[dict objectForKey:[planets objectAtIndex:p]] objectForKey:@"aspects"];
                                [entryDict setObject:entry forKey:date0];
                                //printf("%f: is=%f last=%f closest=%f\n",
                                //       target[p][p1][i], is, lastAngle[p][p1][i], closestAngle[p][p1][i]);
                                //printf("* * * %s * * *\n", [[entry description] cString]);
                                closestAngle[p][p1][i] = 1000.0;
                            }
                            lastAngle[p][p1][i] = is;
                        }
                    } // end angles
                    if (featureMirrors)
                    {   double  m = mirror(deg, deg1);

                        for (i=divCnt-1; i>=0; i--)	// use the biggest divider, if more than one divider fit
                        {   double	is = m - floor(m / target[p][p1][i]) * target[p][p1][i];

                            if ( Diff(is, target[p][p1][i]) <= Diff(lastMirror[p][p1][i], target[p][p1][i]) )
                                closestMirror[p][p1][i] = is;
                            else if (closestMirror[p][p1][i] < 360.0
                                     && Diff(closestMirror[p][p1][i], target[p][p1][i]) < 3.0)
                            {   double			v, delta, delta1;
                                int			dSec;
                                NSMutableDictionary	*entryDict;
                                NSArray			*entry;

                                /* get exact date: date += deltaTime * delta1 / delta */
                                v = (is < closestMirror[p][p1][i]) ? is+target[p][p1][i] : is;
                                delta  = Diff(closestMirror[p][p1][i], target[p][p1][i]) +
                                         Diff(v, target[p][p1][i]);
                                delta1 = Diff(closestMirror[p][p1][i], target[p][p1][i]);
                                dSec = minutes*60 * delta1 / delta  - (minutes*60.0);
                                date0 = [date dateByAddingYears:0 months:0 days:0 hours:0 minutes:dSec/60 seconds:0];

                                v = floor((m+target[p][p1][i]/2.0)/target[p][p1][i])*target[p][p1][i];
                                if (v >= 360.0) v = 0.0;
                                entry = [NSArray arrayWithObjects:[planets objectAtIndex:p1],
                                        [dividers objectAtIndex:i], [NSNumber numberWithDouble:v], @"Mirror", nil];
                                entryDict = [[dict objectForKey:[planets objectAtIndex:p]] objectForKey:@"aspects"];
                                [entryDict setObject:entry forKey:date0];
                                closestMirror[p][p1][i] = 1000.0;
                            }
                            lastMirror[p][p1][i] = is;
                        }
                    } // end mirrors
                }
            }
        } // end aspects

        date = [date dateByAddingYears:0 months:0 days:0 hours:0 minutes:minutes seconds:0];
    }

NSLog(@"Search dict = %@", dict);
    return dict;
}


/* search aspects
 * created:  2004-05-20
 * modified: 2009-02-22 (find flip via flipping modulo(angle) from both sides, don't include approaches)
 *           2006-05-23 (progress bar added)
 *
 * return
 * {
 *    date = (planet1, planet2, divider, angle);
 *    date = (planet1, planet2, divider, mirror, "Mirror");
 * }
 */
- (NSDictionary*)searchAspect:(NSString*)planet1 :(NSString*)planet2
                     dividers:(NSArray*)dividers
                 searchAngles:(BOOL)searchAngles searchMirrors:(BOOL)searchMirrors
                     fromDate:(NSCalendarDate*)dateFrom toDate:(NSCalendarDate*)dateTo
               searchInterval:(int)minutes
{   NSMutableDictionary *dict = [NSMutableDictionary dictionary];
    NSCalendarDate      *date = dateFrom, *date0;
    int                 p1 = [ephemeris indexForObjectName:planet1];
    int                 p2 = [ephemeris indexForObjectName:planet2];
    int                 i, divCnt = [dividers count], loop, loopCnt;
    double              target[divCnt], lastAngle[divCnt], closestAngle[divCnt];
    double              lastMirror[divCnt], closestMirror[divCnt];
    NSArray             *entry;
    EphemerisFlags      ephFlags = EPHFLAG_LON; // what to calculate
    EphemerisObjects    ephObjIx[2];            // objects to calculate
    int                 ephObjCnt = 0;          // number of objects in array

    if (!searchAngles && !searchMirrors)
        return dict;
    dividers = [dividers sortedArrayUsingSelector:@selector(compare:)];
    for (i=0; i<divCnt; i++)
    {   target[i] = 360.0 / [[dividers objectAtIndex:i] floatValue];
        lastAngle[i]  = closestAngle[i]  = 1000.0;
        lastMirror[i] = closestMirror[i] = 1000.0;
    }

    if (p1 != EPH_NONE)
        ephObjIx[ephObjCnt++] = p1;
    if (p2 != EPH_NONE)
        ephObjIx[ephObjCnt++] = p2;

    loop = 0;
    loopCnt = ([dateTo timeIntervalSinceDate:date]/60.0) / minutes;
    while ([date compare:dateTo] != NSOrderedDescending)	// date <= dateTo
    {   NSAutoreleasePool   *pool = [NSAutoreleasePool new];
        EphObject           *objects;	// = [ephemeris objectsAtUTC:date];

        objects = [ephemeris objectsAtJD:[ephemeris julianDayForDate:date bc:NO] flags:ephFlags
                                 objects:ephObjIx cnt:ephObjCnt
                               longitude:topoLon latitude:topoLat elevation:topoElev
                             topocentric:topocentricSearch];

        [date retain];
        if (searchAngles)
        {   double	a = angle(objects[p1].lon, objects[p2].lon);

            for (i=divCnt-1; i>=0; i--)	// use the biggest divider, if more than one divider fit
            {   double	is = a - floor(a / target[i]) * target[i];      // modulo denom
                double  isDelta = Min(is, target[i] - is);              // delta from target
                double  lsDelta = Min(lastAngle[i], target[i] - is);    // delta from target

                //printf("%s: last=%f is=%f d=%f closest=%f t=%f\n", [[date  descriptionWithCalendarFormat:@"%Y%m%d %H:%M:%S"] cString], lastAngle[i], is, isDelta, closestAngle[i], target[i]);

                if ( Diff(lastAngle[i], is) < target[i]/2.0 &&  // no flip
                     isDelta <= lsDelta )                       // delta getting smaller
                     //Diff(is, target[i]) <= Diff(lastAngle[i], target[i]) )	// delta getting smaller (one direction only!)
                    closestAngle[i] = is;
                else if ( closestAngle[i] < 500.0 && (Diff(lastAngle[i], is) > target[i]/2.0    // flip
                          /*|| Diff(closestAngle[i], target[i]) < 1.0*/) )                      // approach
                {   double	isSign = ((is           > target[i]/2.0) ? -(target[i]-is) : is);
                    double	prSign = ((lastAngle[i] > target[i]/2.0) ? -(target[i]-lastAngle[i]) : lastAngle[i]);
                    //int	dir = (isSign > prSign) ? 1 : -1;
                    double	v, delta, delta1;
                    int		dSec;

                    /* get exact date: date += deltaTime * delta1 / delta */
                    if (isSign < prSign + 0.00001)   // retrograde -> we are one step to far (no, fixed)
                    {
                        delta  = Diff(isSign, prSign);  // delta of time interval
                        delta1 = Abs(prSign);           // delta to target
                        dSec = - ((minutes*60) - (minutes*60 * delta1 / delta));
                        if (Abs(dSec/60) > minutes)
                        {   NSLog(@"searchAspect, error in calculation: dSec = %f", dSec);
                            dSec = 0.0;
                        }
                        date0 = [date dateByAddingYears:0 months:0 days:0 hours:0 minutes:dSec/60 seconds:0];
                    }
                    else	// FIXME: this should work with the signed angles as well
                    {
                        v = (is < closestAngle[i]) ? is+target[i] : is;
                        delta  = Diff(closestAngle[i], target[i]) + Diff(v, target[i]);
                        delta1 = Diff(closestAngle[i], target[i]);
                        dSec = minutes*60 * delta1 / delta - (minutes*60.0);
                        date0 = [date dateByAddingYears:0 months:0 days:0 hours:0 minutes:dSec/60 seconds:0];
                    }

                    /*printf("last=%f (%f) is=%f (%f) closest=%f t+is=%f t=%f\n", lastAngle[i], lsDelta, is, isDelta, closestAngle[i], is+target[i], target[i]);
                    printf("date='%s' date0='%s'  v=%f delta=%f delta1=%f\n",
                        [[date  descriptionWithCalendarFormat:@"%Y%m%d %H:%M:%S"] cString],
                        [[date0 descriptionWithCalendarFormat:@"%Y%m%d %H:%M:%S"] cString], v, delta, delta1);*/

                    v = floor((a+target[i]/2.0)/target[i])*target[i];	// target angle
                    entry = [NSArray arrayWithObjects:planet1, planet2,
                                                     [dividers objectAtIndex:i],
                                                     [NSNumber numberWithDouble:v],
                                                     nil];
                    [dict setObject:entry forKey:date0];
                    //printf("%f: is=%f last=%f closest=%f\n", target[i], is, lastAngle[i], closestAngle[i]);
                    //printf("* * * %s * * *\n", [[entry description] cString]);
                    closestAngle[i] = 1000.0;
                }
                lastAngle[i] = is;
            }
        }

        if (searchMirrors)
        {   double	m = mirror(objects[p1].lon, objects[p2].lon);

            for (i=0; i<divCnt; i++)
            {   double	is = m - floor(m / target[i]) * target[i];

                // FIXME: we should ignore mirror angles covered by higher dividers !!!
                if ( Diff(is, target[i]) <= Diff(lastMirror[i], target[i]) )
                    closestMirror[i] = is;
                else if (closestMirror[i] < 500.0 && Diff(closestMirror[i], target[i]) < 1.0)
                {   double	isSig = ((is            > target[i]/2.0) ? -(target[i]-is) : is);
                    double	prSig = ((lastMirror[i] > target[i]/2.0) ? -(target[i]-lastMirror[i])
                                                                     : lastMirror[i]);
                    double	v, delta, delta1;
                    int		dSec;

                    /* get exact date: date += deltaTime * delta1 / delta */
                    if (isSig < prSig + 0.00001)    // retrograde -> we are one step to far
                    {
                        delta  = Diff(isSig, prSig);	// delta of time interval
                        delta1 = Abs(prSig);		// delta to target
                        dSec = - (minutes*60 * delta1 / delta) - (minutes*60);
                        date0 = [date dateByAddingYears:0 months:0 days:0 hours:0 minutes:dSec/60 seconds:0];
                    }
                    else	// FIXME: this should work with the signed angles as well
                    {
                        v = (is < closestMirror[i]) ? is+target[i] : is;
                        delta  = Diff(closestMirror[i], target[i]) + Diff(v, target[i]);
                        delta1 = Diff(closestMirror[i], target[i]);
                        dSec = minutes*60 * delta1 / delta  - (minutes*60.0);
                        date0 = [date dateByAddingYears:0 months:0 days:0 hours:0 minutes:dSec/60 seconds:0];
                    }
                    v = floor((m+target[i]/2.0)/target[i])*target[i];	// target mirror angle
                    entry = [NSArray arrayWithObjects:planet1, planet2,
                                                      [dividers objectAtIndex:i],
                                                      [NSNumber numberWithDouble:v],
                                                      @"Mirror", nil];
                    [dict setObject:entry forKey:date0];
                    //printf("%f: is=%f last=%f closest=%f\n", target[i], is, lastMirror[i], closestMirror[i]);
                    //printf("* * * %s * * *\n", [[entry description] cString]);
                    closestMirror[i] = 1000.0;
                }
                lastMirror[i] = is;
            }
        }

        [progressTarget performSelector:progressAction
                             withObject:[NSNumber numberWithFloat:(float)loop++/(float)loopCnt]];

        [date autorelease];
        date = [[date dateByAddingYears:0 months:0 days:0 hours:0 minutes:minutes seconds:0] retain];
        [pool release];
    }

    [progressTarget performSelector:progressAction
                         withObject:[NSNumber numberWithFloat:1.0]];
    [progressTarget performSelector:@selector(setTitle:) withObject:nil];

//NSLog(@"Search dict = %@", dict);

    return dict;
}

@end
