/* astroCommon.m
 * Common astronomical functions
 *
 * Copyright (C) 1998-2015 by Cenon GmbH
 * Author:   Georg Fleischmann
 *
 * created:  1998-11-06
 * modified: 2015-12-30 (sunNode(): float->double)
 *           2015-06-07 (cenJDFromNSDate(), cenIntDateFromJD() added)
 *           2015-03-30 (vhfJDFromNSDate() calendar released)
 *           2009-04-08 (modulo())
 *           2006-04-06 (Pholus ']' added, Pallas '[' corrected, new astro functions without NSCalendarDate)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the Cenon Public License as
 * published by Cenon GmbH. Among other things, the
 * License requires that the copyright notices and this notice
 * be preserved on all copies.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the Cenon Public License for more details.
 *
 * You should have received a copy of the Cenon Public License along
 * with this program; see the file LICENSE. If not, write to Cenon.
 *
 * Cenon GmbH, Schwabstr. 45, 72108 Rottenburg a.N., Germany
 * eMail: info@cenon.de
 * http://www.cenon.info
 */

#include <Foundation/Foundation.h>
#include <math.h>
#include <VHFShared/types.h>
#include <VHFShared/VHFStringAdditions.h>
#include "astroCommon.h"

#define COS_E 0.9175237	// Cos(23.4333)
#define SIN_E 0.3976812	// Sin(23.4333)

/* ascending Sun's Node
 * Intersection between the Sun's equator and the ecliptic.
 * the descending node is always 180 degree to the ascending node.
 */
double sunNode(int year)
{
    return 73.66670 + (double)(year - 1850) * 0.01396;   // 2014 = 75.956, 2016 = 75.98406
    //return 75.76  + (double)(year - 2000) * 0.01397;   // 2014 = 75.955
}

/* ascending invariable node
 * Intersection between the invarianle plane and the ecliptic.
 */
double invariableNode(int year)
{
    //return 107.5822;    // J2000
    return 102.95833 + (double)(year - 1750) * 0.01849548;  //1750 = 102.95833, 2016 = 107.878
}


NSMutableData* reducedTable(NSData *tableData, int tableSize, double factor)
{   int             i;
    NSMutableData   *nTableData = [NSMutableData dataWithLength:tableSize * sizeof(double)];
    double          *table = (double*)[tableData bytes], *nTable = [nTableData mutableBytes];

    /* reduce  */
    for ( i=0; i<tableSize; i++ )
        nTable[i] = table[i] * factor;
    return nTableData;
}

/* [0, 1] or [-1, 1]
 */
NSMutableData* normalizedTable(NSData *tableData, int tableSize, BOOL positive)
{   int			i;
    double		min, max, absMax;
    NSMutableData	*nTableData = [NSMutableData dataWithLength:tableSize * sizeof(double)];
    double		*table = (double*)[tableData bytes], *nTable = [nTableData mutableBytes];

    /* get max, sum */
    for ( i=0, min=LARGE_COORD, max=LARGENEG_COORD; i<tableSize; i++ )
    {
        min = Min(table[i], min);
        max = Max(table[i], max);
    }

    absMax = Max(Abs(max), Abs(min));
    if (absMax == 0.0)
        absMax = 1.0;

    /* normalize  */
    for ( i=0; i<tableSize; i++ )
    {
        if (positive)
        {
            nTable[i] = table[i] - min;			// move to 0
            if (max-min)
                nTable[i] = nTable[i] * 1.0 / (max-min);	// [0, 1]
        }
        else
            nTable[i] = table[i] / absMax;		// [-1, 1]
    }
    return nTableData;
}

/*
 * 1 2 3 4 ...
 *   2   4 ...
 *     3   ...
 *       4 ...
 *
 * range = [0, 1]
 */
double *harmonicTable(int tableSize)
{   double	*table = NSZoneMalloc(NULL, tableSize * sizeof(double));
    int		i, j, a;

    for ( i=0; i<tableSize; i++ )
        table[i] = 0.0;

    for ( i=1; i<=tableSize; i++ )
    {   int	cnt = 0;

        for ( j=1; j<=i; j++ )
            if ( !(i % j) )
                cnt++;
        //table[i-1] = (double)cnt / (double)i;		// numbers
        /* set angles */
        for ( a=tableSize/i; a<=tableSize; a+=tableSize/i )
        {   int	a1 = (a>=tableSize) ? 0 : a;
            if ( !table[a1] )
                table[a1] = (double)cnt / (double)i;
        }
    }

    return table;
}

/* linear function (analog sinus)
 *   0  0
 *  90  1
 * 180  0
 * 270 -1
 */
double lin(double v)
{
    while (v >= 360.0)
        v -= 360.0;
    if (v <= 90.0)
        return 0 + v / 90.0;
    if (v <= 180.0)
        return 1 - (v-90) / 90.0;
    if (v <= 270.0)
        return 0 - (v-180) / 90.0;
    return -1 + (v-270) / 90.0;
}
/* calculate harmonic pattern
 * this is the sum of the harmonics of a longitudinal wave
 * range = [-1, 1]
 */
double *harmonicPattern(int tableSize, int depth)
{   double	*table = NSZoneMalloc(NULL, tableSize * sizeof(double));
    double	max;
    int		d, i, t;

    //for ( d=0.0; d<tableSize; d+=1.0)
    //    table[(int)d] = harmonicValue(d*360.0/(double)tableSize);

//#if 0
    for ( i=0; i<tableSize; i++ )
        table[i] = 0.0;

    /* basic curve */
    //for ( t=0; t<tableSize; t+=1 )
    //    table[t] += Sin(t);

    /* harmonics: integer deviders 1, 2, 3, 4, 5, ...
     * fundamental and every harmonic is devided recursively
     */
    /*for ( d=1; d<=depth; d*=2 )		// harmonics
    {
        for ( t=0; t<tableSize; t+=1 )		// table column
            table[t] += Abs(Sin(t * d * 360.0/tableSize));
    }*/

    /* harmonics: integer deviders 1, 2, 3, 4, 5, ... */
    for ( d=1; d<=depth; d+=1 )			// harmonic
    {
        for ( t=0; t<tableSize; t+=1 )		// table column
            table[t] += Abs(Sin(t * (double)d * 360.0/tableSize)) /*/ d*/;
    }

    /* get max, sum */
    for ( i=0, max=0.0; i<tableSize; i++ )
        max = Max(Abs(table[i]), max);

    /* normalize, invert  */
    for ( i=0; i<tableSize; i++ )
        table[i] = 1.0 - (table[i] / max);	// [0, 1]
//#endif

    /* invert 2nd part
     */
    /*for ( i=0; i<tableSize; i++ )
    {
        if (i<=tableSize/2)
            table[i] *= 0.5;
        else
            table[i] = (1.0 - table[i] + 1.0) * 0.5;
    }*/

    /* add basic curve, and normalize again */
    /*for ( t=0; t<tableSize; t+=1 )
        table[t] = Sin(t) + table[t]/2.0;
    for ( i=0, max=0.0; i<tableSize; i++ )
        max = Max(Abs(table[i]), max);
    for ( i=0; i<tableSize; i++ )
        table[i] = table[i] / max;*/		// [-1, 1]

    return table;
}

/* deg = [0, 360]
 */
#define HARMONIC_RES	360.0
double harmonicValue(double deg)
{   double	d /*, max = 5.88104575996997259*/ /*4.414657969489995*/, v = 0.0;

    /* harmonics: integer deviders 1, 2, 3, 4, 5, ... */
    for ( d=1.0; d<=HARMONIC_RES; d+=1.0 )	// harmonic
    {   double	waveLen = 360.0 / d;

        v += Sin((deg-floor(deg/waveLen)*waveLen) / 2.0 * d);
        //v += Abs(Sin(deg/0.5 * d)) / d;
    }
    v = 1.0 - (v/HARMONIC_RES);			// normalize, invert
    return v;
}

/* calculate harmonic pattern
 * 360 - 180 -> 120, 240
 * 180 -  90 ->  60, 300
 *  90 -  45 ->  30, 330, 150, 210
 * range = [-1, 1]
 */
double *harmonicPattern1(int tableSize)
{   double	*table = NSZoneMalloc(NULL, tableSize * sizeof(double));
    double	o, lastO = 0.0;
    int		i;

    for ( i=0; i<tableSize; i++ )
        table[i] = 0.0;

    /* octaves */
    for ( o=360.0; o>=360.0/(double)tableSize; o/=2.0 )
    {
        table[(o>=360.0) ? 0 : (int)(o*(double)tableSize/360.0)] = 1.0 * o/360.0;	// octave
        if ( o<360.0 )
        {   double	h = (lastO - o) / 3.0, seg;

            for ( seg = o; seg<360.0; seg += lastO )
            {
                if ( table[(int)(seg*(double)tableSize/360.0)] == 0.0 )
                    table[(int)(seg*(double)tableSize/360.0)] = 1.0 * o/360.0;	// octave
                if ( table[(int)(seg+h)] == 0.0 )
                    table[(int)(seg+h)] = -1.0 * o/360.0;
                if ( table[(int)(seg-h)] == 0.0 )
                    table[(int)(seg-h)] = -1.0 * o/360.0;
            }
        }
        lastO = o;
    }

    return table;
}

/* revert angles more than 180 degrees from ref
 * a    ref
 * -10  0   -> -10
 * -185 0   -> 175
 */
/*float comparableAngle180( float a, float ref)
{
    if ( a < ref && a < ref-180.0 )
        a += 360.0;
    else if ( a > ref && a > ref+180.0 )
        a -= 360.0;
    return a;
}*/

/* equal angles
 * Angle from sun to moon SunMoon: deg1=Sun, deg2=Moon
 */
/*float angleFromFirstToSecond( float deg1, float deg2 )
{   float	a;

    if ( deg1 < deg2 )
        deg1 += 360.0;
    a = deg2 - deg1;
    if ( a > 180.0 )
        a -= 360.0;
    else if ( a < -180.0 )
        a += 360.0;
    return a;
}*/

/*float angleBetween( float deg1, float deg2 )
{   float	angle;

    angle = Diff(deg1, deg2);
    if ( angle > 180.0)
        angle = 360.0 - angle;
    return angle;
}*/

/* 137.1234 -> 137 7'24"
 * moved to AstroPrincipal to localize degreee symbol !
 */
/*NSString *stringFromDeg(float deg)
{   int		d, m, s;
    NSString	*string;

    d = (int)deg;
    m = (int)floor((deg-d) * 60.0);
    s = (int)floor((deg-d-m/60.0) * 3600.0 + 0.5);
    string = [NSString stringWithFormat:@"%@ %@'%@\"",
                       [[NSString stringWithFormat:@"%d", d] stringWithLength:-4],
                       [[NSString stringWithFormat:@"%d", m] stringWithLength:-2],
                       [[NSString stringWithFormat:@"%d", s] stringWithLength:-2] ];
    return string;
}*/

/* "430'00" -> 4.5 or 4.5
 */
float degFromString(NSString *degString)
{   float           deg = 0.0;
    NSScanner       *scanner = [NSScanner scannerWithString:degString];
    NSCharacterSet  *digitSet  = [NSCharacterSet decimalDigitCharacterSet];
    NSCharacterSet  *degreeSet = [NSCharacterSet characterSetWithCharactersInString:@" .'\""];
    NSString        *string;
    NSRange         range;

    range = [degString rangeOfCharacterFromSet:degreeSet];
    if ( !range.length )	// 4.5
        deg = [degString floatValue];
    else			// 4 30'00"
    {
        if ( [scanner scanCharactersFromSet:digitSet intoString:&string] )
            deg += [string floatValue];
        [scanner scanUpToCharactersFromSet:digitSet intoString:NULL];
        if ( [scanner scanCharactersFromSet:digitSet intoString:&string] )
            deg += [string floatValue] / 60.0;
        [scanner scanUpToCharactersFromSet:digitSet intoString:NULL];
        if ( [scanner scanCharactersFromSet:digitSet intoString:&string] )
            deg += [string floatValue] / 3600.0;
    }
    return deg;
}

/* created: 1997-08-21
 */
NSInteger sortAsNumbers(id str1, id str2, void *context)
{
    if ( [str1 intValue] < [str2 intValue] )
        return NSOrderedAscending;
    if ( [str1 intValue] > [str2 intValue] )
        return NSOrderedDescending;
    return NSOrderedSame;
}

/* MJD: modified julian day (starting in 1858)
 */
double vhfMJD( NSCalendarDate *date )
{   int		year, month, day;
    double	hour, mjd;
    double	a, b;

    year  = [date yearOfCommonEra];
    month = [date monthOfYear];
    day   = [date dayOfMonth];
    hour  = (double)[date hourOfDay]+[date minuteOfHour]/60.0+[date secondOfMinute]/3600.0;

    a = 10000.0*year + 100.0*month + day;
    if ( month <= 2 )
    {   month += 12;
        year--;
    }
    if ( a <= 15821004.1 )
        b = -2.0+floor((year+4716.0)/4.0)-1179.0;
    else
        b = floor(year/400.0)-floor(year/100.0)+floor(year/4.0);
    a = 365.0*year-679004.0;
    mjd = a+b+floor(30.6001*(month+1.0))+day+hour/24.0;
    return mjd;
}

NSCalendarDate *dateFromMJD(double mjd)
{   int			year, month, day, hour, min, sec;
    int			b, d, f;
    double		jd, jd0, c, e;
    NSCalendarDate	*date;

    jd = mjd+2400000.5;
    jd0 = floor(jd+0.5);
    if ( jd0<2299161.0 )
        c = jd0+1524.0;
    else
    {
        b = floor((jd0-1867216.25)/36524.25);
        c = jd0+(b-floor(b/4.0))+1525.0;
    }
    d = floor((c-122.1)/365.25);
    e = 365.0*d+floor(d/4.0);
    f = floor((c-e)/30.6001);
    day = floor(c-e+0.5)-floor(30.6001*f);
    month = f-1-12*floor(f/14.0);
    year = d-4715-floor((7.0+month)/10.0);
    hour = 24.0*(jd+0.5-jd0);
    min = (24.0*(jd+0.5-jd0) - hour)/60.0;
    sec = ((24.0*(jd+0.5-jd0) - hour) - min*60.0)/3600.0;

    date = [NSCalendarDate dateWithYear:year month:month day:day hour:hour minute:min second:sec
                               timeZone:[NSTimeZone timeZoneForSecondsFromGMT:0]];
    return date;
}

double vhfSideralTimeMJD(double mjd, float lon)
{   double	mj = mjd, mjd0, t, ut, gmst, lmst;

    mjd0 = floor( mj );
    ut = (mj-mjd0)*24;
    t = (mjd0-51544.5)/36525.0;
    gmst = 6.697374558+1.0027379093*ut+(8640184.812866+(0.093104-6.2e-6*t)*t)*t/3600.0;
    lmst = 24.0*frac((gmst+lon/15.0)/24.0);
    return lmst;	// hours
}


/* Julian day from NSCalendarDate (starting -4712)
 */
double cenJD(int year, int month, int day, int H, int M, int S)
{   double	hour  = (double)H+M/60.0+S/3600.0;
    double  y0, y1, jd;

    if ( month <= 2 )
    {   month += 12;
        year--;
    }
    y0 = year;
    jd = floor((y0+4712.0)*365.25) + floor(30.6*(month+1.0)+0.000001) + day + hour/24.0 - 63.5;

    y1 = Abs(year)/100 - Abs(year)/400;
    if (year < 0)
        y1 = -y1;
    jd = jd - y1 + 2.0;
    if ((year < 0) && (y0/100.0 == floor(y0/100.0)) && (y0/400.0 != floor(y0/400.0)))
        jd -= 1.0;

    return jd;
}

/* Julian day from NSDate (starting -4712)
 */
double vhfJDFromNSDate(NSDate *date)    { return cenJDFromNSDate(date); }
double cenJDFromNSDate(NSDate *date)
{   NSCalendar          *calendar = [[NSCalendar alloc] initWithCalendarIdentifier:NSGregorianCalendar];
    unsigned            unitFlags = NSYearCalendarUnit | NSMonthCalendarUnit |  NSDayCalendarUnit
                                  | NSHourCalendarUnit | NSMinuteCalendarUnit| NSSecondCalendarUnit;
    NSDateComponents    *dateParts;
    int                 year, month, day;
    double              hour, y0, y1, jd;

    [calendar setTimeZone:[NSTimeZone timeZoneForSecondsFromGMT:0]];    // we want UTC
    dateParts = [calendar components:unitFlags fromDate:date];
    year  = [dateParts year];
    month = [dateParts month];
    day   = [dateParts day];
    hour  = (double)[dateParts hour]+[dateParts minute]/60.0+[dateParts second]/3600.0;

    if ( month <= 2 )
    {   month += 12;
        year--;
    }
    y0 = year;
    jd = floor((y0+4712.0)*365.25) + floor(30.6*(month+1.0)+0.000001) + day + hour/24.0 - 63.5;

    y1 = Abs(year)/100 - Abs(year)/400;
    if (year < 0)
        y1 = -y1;
    jd = jd - y1 + 2.0;
    if ((year < 0) && (y0/100.0 == floor(y0/100.0)) && (y0/400.0 != floor(y0/400.0)))
        jd -= 1.0;

    [calendar release];
    return jd;
}

/* Julian day from NSCalendarDate (starting -4712)
 */
double vhfJulianDay(NSCalendarDate *date)
{   int		year, month, day;
    double	hour, y0, y1, jd;

    year  = [date yearOfCommonEra];
    month = [date monthOfYear];
    day   = [date dayOfMonth];
    hour  = (double)[date hourOfDay]+[date minuteOfHour]/60.0+[date secondOfMinute]/3600.0;

    if ( month <= 2 )
    {   month += 12;
        year--;
    }
    y0 = year;
    jd = floor((y0+4712.0)*365.25) + floor(30.6*(month+1.0)+0.000001) + day + hour/24.0 - 63.5;

    y1 = Abs(year)/100 - Abs(year)/400;
    if (year < 0)
        y1 = -y1;
    jd = jd - y1 + 2.0;
    if ((year < 0) && (y0/100.0 == floor(y0/100.0)) && (y0/400.0 != floor(y0/400.0)))
        jd -= 1.0;

    return jd;
}

/* Julian day (starting -4712)
 * modified: 2008-10-16 (hour had integer conversion error = too big)
 */
double vhfJulianDayFromInt(int yyyymmdd, int hhmm)
{   int		year, month, day;
    double	hour, y0, y1, jd;

    year  = yyyymmdd / 10000;
    yyyymmdd -= yyyymmdd/10000*10000;
    month = yyyymmdd / 100;
    yyyymmdd -= yyyymmdd/100*100;
    day   = yyyymmdd;
    hour  = (double)(hhmm/100) + (double)(hhmm-(hhmm/100*100)) / 60.0;

    if ( month <= 2 )
    {   month += 12;
        year--;
    }
    y0 = year;
    jd = floor((y0+4712.0)*365.25) + floor(30.6*(month+1.0)+0.000001) + day + hour/24.0 - 63.5;

    y1 = Abs(year)/100 - Abs(year)/400;
    if (year < 0)
        y1 = -y1;
    jd = jd - y1 + 2.0;
    if ((year < 0) && (y0/100.0 == floor(y0/100.0)) && (y0/400.0 != floor(y0/400.0)))
        jd -= 1.0;

    return jd;
}
/* date from Julian day (starting -4712)
 */
int cenIntDateFromJulianDay(double jd, int *yymmdd, int *hhmm) { return cenIntDateFromJD(jd, yymmdd, hhmm); }
int vhfIntDateFromJulianDay(double jd, int *yymmdd, int *hhmm) { return cenIntDateFromJD(jd, yymmdd, hhmm); }   // DEPRECATED
int cenIntDateFromJD (double jd, int *yyyymmdd, int *hhmm)
{   int		year, month, day, yymmdd;
    double	u0, u1, u2, u3, u4, hour;

    u0 = jd + 32082.5;
    u1 = u0 + floor(u0/36525.0) - floor(u0/146100.0) - 38.0;
    if (jd >= 1830691.5)
        u1 += 1;
    u0 = u0 + floor(u1/36525.0) - floor(u1/146100.0) - 38.0;

    u2 = floor(u0 + 123.0);
    u3 = floor((u2 - 122.2) / 365.25);
    u4 = floor((u2 - floor(365.25 * u3) ) / 30.6001);
    month = (int)(u4 - 1.0);
    if (month > 12)
        month -= 12;
    day   = (int) (u2 - floor(365.25 * u3) - floor(30.6001 * u4));
    year  = (int) (u3 + floor((u4 - 2.0) / 12.0) - 4800);
    hour  = (jd - floor(jd + 0.5) + 0.5) * 24.0;

    yymmdd = year*10000 + month*100 + day;
    if (yyyymmdd)
        *yyyymmdd = yymmdd;
    if (hhmm)
        *hhmm     = (int)hour*100 + (int)((hour-floor(hour))*60.0);
    return yymmdd;
}

NSDate *cenNSDateFromJulianDay(double jd) { return cenNSDateFromJD(jd); }
NSDate *cenNSDateFromJD(double jd)
{   int		y, m, d, H, M, S;
    double	u0, u1, u2, u3, u4, hour;

    u0 = jd + 32082.5;
    u1 = u0 + floor(u0/36525.0) - floor(u0/146100.0) - 38.0;
    if (jd >= 1830691.5)
        u1 += 1;
    u0 = u0 + floor(u1/36525.0) - floor(u1/146100.0) - 38.0;

    u2 = floor(u0 + 123.0);
    u3 = floor((u2 - 122.2) / 365.25);
    u4 = floor((u2 - floor(365.25 * u3) ) / 30.6001);
    m = (int)(u4 - 1.0);
    if (m > 12)
        m -= 12;
    d = (int) (u2 - floor(365.25 * u3) - floor(30.6001 * u4));
    y = (int) (u3 + floor((u4 - 2.0) / 12.0) - 4800);
    hour = (jd - floor(jd + 0.5) + 0.5) * 24.0;
    H = (int)hour;
    M = (int)((hour-(double)H)*60.0);
    S = (int)((hour-(double)H-(double)M/60.0)*3600.0);

    return [NSDate dateWithString:[NSString stringWithFormat:@"%d-%s%d-%s%d %s%d:%s%d:%s%d +0000",
                                   y, (m<10)?"0":"", m, (d<10)?"0":"", d,
                                   (H<10)?"0":"", H, (M<10)?"0":"", M, (S<10)?"0":"", S]];
}
NSCalendarDate *vhfNSDateFromJD(double jd)
{   int		y, m, d, H, M, S;
    double	u0, u1, u2, u3, u4, hour;

    u0 = jd + 32082.5;
    u1 = u0 + floor(u0/36525.0) - floor(u0/146100.0) - 38.0;
    if (jd >= 1830691.5)
        u1 += 1;
    u0 = u0 + floor(u1/36525.0) - floor(u1/146100.0) - 38.0;

    u2 = floor(u0 + 123.0);
    u3 = floor((u2 - 122.2) / 365.25);
    u4 = floor((u2 - floor(365.25 * u3) ) / 30.6001);
    m = (int)(u4 - 1.0);
    if (m > 12)
        m -= 12;
    d = (int) (u2 - floor(365.25 * u3) - floor(30.6001 * u4));
    y = (int) (u3 + floor((u4 - 2.0) / 12.0) - 4800);
    hour = (jd - floor(jd + 0.5) + 0.5) * 24.0;
    H = (int)hour;
    M = (int)((hour-(double)H)*60.0);
    S = (int)((hour-(double)H-(double)M/60.0)*3600.0);

    return [NSCalendarDate dateWithString:[NSString stringWithFormat:@"%d-%s%d-%s%d %s%d:%s%d:%s%d +0000",
                                   y, (m<10)?"0":"", m, (d<10)?"0":"", d,
                                   (H<10)?"0":"", H, (M<10)?"0":"", M, (S<10)?"0":"", S]];
}

/* gmt to local time
 * start at 22.09. 00:00 UTC
 * each 24h (86400s) +4 minutes (star day = 23:56:04.1) -> +235.9 sec
 * //we also add a correction of 9.8s per hour of day
 */
float frac(float x)
{
    x -= floor(x);
    if (x<0) x += 1.0;
    return x;
}


/* sideral time in hours with comma (reference date = -4712)
 */
double vhfSideralTime(double jd, float lon)
{   double	mj = jd, mjd0, t, ut, gmst, lmst;

    mjd0 = floor(mj);
    ut = (mj-mjd0-0.5) * 24.0;
    t = (mjd0-2451544.5) / 36525.0;
    gmst = 6.697374558+1.0027379093*ut+(8640184.812866+(0.093104-6.2e-6*t)*t)*t/3600.0;
    lmst = 24.0*frac((gmst+lon/15.0)/24.0);
    return lmst;	// hours
}

float vhfLonForSideralTime(double st, NSCalendarDate *gmt)
{   double	mj, mjd0, t, ut, gmst, lon;

    mj = vhfMJD(gmt);
    mjd0 = floor(mj);
    ut = (mj-mjd0)*24;
    t = (mjd0-51544.5)/36525.0;
    gmst = 6.697374558+1.0027379093*ut+(8640184.812866+(0.093104-6.2e-6*t)*t)*t/3600.0;
    lon = (st - gmst) * 15.0;

    while (lon <= -180.0)
        lon += 360.0;
    while (lon > 180.0)
        lon -= 360.0;

    return lon;
}

double vhfRA(double st)
{
    return st * 15.0;
}

double vhfMC(double st)
{   double	l, mc, ra;

    ra = st * 15.0;	// secs / (4.0*60.0);
    l = COS_E * Cot(ra);
    mc = Acot(l);
    if ( mc < 0.0 )
        mc += 360.0;

    /* from 12:00 - 00:00 sideralTime -> mc = 180 - 360 degree
     * from 00:00 - 12:00 sideralTime -> mc = 0 - 180 degree
     * -06:00 -> mc
     * 06:00- -> mc -= 180.0
     */
    if ( st >= 0.0 && st < 12.0 )
    {
        if ( mc > 180.0 )
            mc -= 180.0;
    }
    else if ( mc < 180.0 )
        mc += 180.0;

    if ( mc < 0.0 )
        mc += 360.0;
    else if ( mc >= 360.0 )
        mc -= 360.0;

    return mc;
}

double vhfAC(double st, float lat)
{   double	oa, ac, m, a;

    oa = st * 15.0 + 90.0;	// secs / (4.0*60.0) + 90.0
    ac  = Acot( COS_E*Cot(oa) - SIN_E*Tan(lat)/Sin(oa) );
    if ( ac < 0 )
        ac += 360.0;

    /* AC is ccw from MC */
    m = vhfMC(st);
    ac = ((a = angle(ac, m)) < 180.0) ? ac : ac+180.0;
    if ( ac >= 360.0 )
        ac -= 360.0;

    /* from 18:00 - 06:00 sideralTime -> ac = 0 - 180 degree
     * from 06:00 - 18:00 sideralTime -> ac = 180 - 360 degree
     * these are the points where all ACs go through, but not the turn around points !!!
     */
    /*if ( s>=21600 && s<64800 )
    {
        if ( ac<180.0 )
            ac += 180.0;
    }
    else if ( ac>180.0 )
        ac -= 180.0;*/

    return ac;
}

/* get latitude for AC or DC degree
 */
BOOL vhfIsAC(float lat, float acDeg, double st)
{   float	a = angle(vhfAC(st, lat), acDeg);

    if ( a < 10.0 || a > 350.0)
        return YES;
    return NO;
}

/* MC + UTC -> sideral time
 * FIXME: we don't need UTC for the sideral time offset (without date) we return
 */
double vhfSideralTimeForMC(float mcDeg, NSCalendarDate *utc)
{   double	l, mc, ra, st;

    mc = mcDeg;
    /*if (mcDeg > 180.0)
        mc -= 180.0;*/

    l = Cot(mc);
    ra = Acot(l / COS_E);
    if (mcDeg >= 90.0 && mcDeg < 270.0)
        ra += 180.0;
    else if (mcDeg >= 270)
        ra += 360.0;
    st = ra * (4.0*60.0) / 3600.0;

    return st;
}

/* the opposite direction for AC/MC
 * MC + UTC -> longitude
 */
float vhfLonForMC(float mcDeg, NSCalendarDate *utc)
{   double	st;
    float	lon;

    st  = vhfSideralTimeForMC(mcDeg, utc);
    lon = vhfLonForSideralTime(st, utc);
    return lon;
}

/* this returns either AC or DC. check later with isAC()
 */
float vhfLatForAC(float acDeg, double st)
{   double	l, oa, lat;

    oa = (st * 3600.0) / (4.0*60.0) + 90.0;
    /* FIXME: if acDeg == 0.0 -> Cot(acDeg) gets infinity */
    l = Cot(acDeg);
    lat = Atan( (COS_E*Cot(oa) - l) * Sin(oa) / SIN_E );

    return lat;
}



/*
 * NSCalendarDate functions (slower), DEPRECATED
 */

NSCalendarDate *sideralTime(NSCalendarDate *gmt, float longitude)
{   double	mj, mjd0, t, ut, gmst, lmst;

    mj = vhfMJD(gmt);
    mjd0 = floor( mj );
    ut = (mj-mjd0)*24;
    t = (mjd0-51544.5)/36525.0;
    gmst = 6.697374558+1.0027379093*ut+(8640184.812866+(0.093104-6.2e-6*t)*t)*t/3600.0;
    lmst = 24.0*frac((gmst+longitude/15.0)/24.0);

    gmt = [NSCalendarDate dateWithString:[gmt descriptionWithCalendarFormat:@"%Y-%m-%d 00:00:00 %z"]];
    //return [gmt dateByAddingTimeInterval:lmst*3600.0];  // 10.6 ...
    gmt = [[[NSCalendarDate alloc] initWithTimeInterval:lmst*3600.0 sinceDate:gmt] autorelease];
    [gmt setTimeZone:[NSTimeZone timeZoneForSecondsFromGMT:0]];
    return gmt;
    //return [gmt dateByAddingYears:0 months:0 days:0 hours:0 minutes:0 seconds:(int)(lmst*3600.0)];
}
/*float longitudeWithSideralTime(NSCalendarDate *st, NSCalendarDate *gmt)
{   double	mj, mjd0, t, ut, gmst, lmst, lon;

    mj = vhfMJD(gmt);
    mjd0 = floor( mj );
    ut = (mj-mjd0)*24;
    t = (mjd0-51544.5)/36525.0;
    gmst = 6.697374558+1.0027379093*ut+(8640184.812866+(0.093104-6.2e-6*t)*t)*t/3600.0;

    gmt = [NSCalendarDate dateWithString: [gmt descriptionWithCalendarFormat:@"%Y%m%d 00:00:00 %z"]
                          calendarFormat:@"%Y%m%d %H:%M:%S %z"];
    lmst = [st timeIntervalSinceDate:gmt] / 3600.0;
    lon = (lmst - gmst) * 15.0;
    while (lon <= -180.0)
        lon += 360.0;
    while (lon > 180.0)
        lon -= 360.0;

    return lon;
}*/

/*float ra( NSCalendarDate *sideralTime)
{   float	s, ra;

    s = [sideralTime timeIntervalSinceDate:[NSCalendarDate dateWithString:[sideralTime descriptionWithCalendarFormat:@"%Y.%m.%d-00:00 %z"] calendarFormat:@"%Y.%m.%d-%H:%M %z"]];
    ra = s / (4.0*60.0);
    return ra;
}*/

float mc( NSCalendarDate *sideralTime)
{   float	s, l, mc, ra;

    s = [sideralTime timeIntervalSinceDate:[NSCalendarDate dateWithString:[sideralTime descriptionWithCalendarFormat:@"%Y.%m.%d-00:00 %z"] calendarFormat:@"%Y.%m.%d-%H:%M %z"]];
    ra = s / (4.0*60.0);
    l = COS_E * Cot(ra);
    mc = Acot(l);
    if ( mc < 0.0 )
        mc += 360.0;

    /* from 12:00 - 00:00 sideralTime -> mc = 180 - 360 degree
     * from 00:00 - 12:00 sideralTime -> mc = 0 - 180 degree
     * -06:00 -> mc
     * 06:00- -> mc -= 180.0
     */
    if ( s>=0 && s<43200 )
    {
        if ( mc > 180.0 )
            mc -= 180.0;
    }
    else if ( mc < 180.0 )
        mc += 180.0;

    if ( mc < 0.0 )
        mc += 360.0;
    else if ( mc >= 360.0 )
        mc -= 360.0;

    return mc;
}
/*static NSCalendarDate *sideralTimeWithMC(float mcDeg, NSCalendarDate *utc)
{   float		s, l, mc, ra;
    NSCalendarDate	*st;

    mc = mcDeg;
    //if (mcDeg > 180.0)
    //    mc -= 180.0;

    l = Cot(mc);
    ra = Acot(l / COS_E);
    if (mcDeg >= 90.0 && mcDeg < 270.0)
        ra += 180.0;
    else if (mcDeg >= 270)
        ra += 360.0;
    s = ra * (4.0*60.0);
    st = [[NSCalendarDate dateWithString:[utc descriptionWithCalendarFormat:@"%Y%m%d 00:00:00 %z"]
                          calendarFormat:@"%Y%m%d %H:%M:%S %z"]
          dateByAddingYears:0 months:0 days:0 hours:0 minutes:0 seconds:s];

    return st;
}*/

float ac( NSCalendarDate *sideralTime, float lat)
{   double	s, oa, ac, m, a;

    s = [sideralTime timeIntervalSinceDate:[NSCalendarDate dateWithString:[sideralTime descriptionWithCalendarFormat:@"%Y.%m.%d-00:00-%z"] calendarFormat:@"%Y.%m.%d-%H:%M-%z"]];
    oa = s / (4.0*60.0) + 90.0;
    ac  = Acot( COS_E*Cot(oa) - SIN_E*Tan(lat)/Sin(oa) );
    if ( ac < 0 )
        ac += 360.0;

    /* AC is ccw from MC */
    m = mc(sideralTime);
    ac = ((a = angle(ac, m)) < 180.0) ? ac : ac+180.0;
    if ( ac >= 360.0 )
        ac -= 360.0;

    /* from 18:00 - 06:00 sideralTime -> ac = 0 - 180 degree
     * from 06:00 - 18:00 sideralTime -> ac = 180 - 360 degree
     * these are the points where all ACs go through, but not the turn around points !!!
     */
    /*if ( s>=21600 && s<64800 )
    {
        if ( ac<180.0 )
            ac += 180.0;
    }
    else if ( ac>180.0 )
        ac -= 180.0;*/

    return ac;
}

/* get latitude for AC or DC degree
 */
/*BOOL isAC(float lat, float acDeg, NSCalendarDate *sideralTime)
{   float	a = angle(ac(sideralTime, lat), acDeg);

    if ( a < 10.0 || a > 350.0)
        return YES;
    return NO;
}*/

/* the opposite direction for AC/MC
 */
/*float longitudeForMC(float mcDeg, NSCalendarDate *utc)
{   NSCalendarDate	*sideralTime;
    float		lon;

    sideralTime = sideralTimeWithMC(mcDeg, utc);
    lon = longitudeWithSideralTime(sideralTime, utc);
    return lon;
}*/
/* this returns either AC or DC. check later with isAC()
 */
/*float latitudeForAC(float acDeg, NSCalendarDate *sideralTime)
{   double	s, l, oa, lat;

    s = [sideralTime timeIntervalSinceDate:[NSCalendarDate dateWithString:[sideralTime descriptionWithCalendarFormat:@"%Y.%m.%d-00:00-%z"] calendarFormat:@"%Y.%m.%d-%H:%M-%z"]];
    oa = s / (4.0*60.0) + 90.0;
    // FIXME: if acDeg == 0.0 -> Cot(acDeg) gets infinity
    l = Cot(acDeg);
    lat = Atan( (COS_E*Cot(oa) - l) * Sin(oa) / SIN_E );

    return lat;
}*/

/* convert planet string to planet symbol
 */
NSString *planetSymbol(NSString *planet)
{
    if ( [planet isEqual:@"Sun"] )          // Sun
        return @"a";
    if ( [planet hasPrefix:@"Mer"] )        // Mer
        return @"b";
    if ( [planet hasPrefix:@"Ven"] )        // Ven
        return @"c";
    if ( [planet hasPrefix:@"Mar"] )        // Mar
        return @"d";
    if ( [planet hasPrefix:@"Jup"] )        // Jup
        return @"e";
    if ( [planet hasPrefix:@"Sat"] )        // Sat
        return @"f";
    if ( [planet hasPrefix:@"Ura"] )        // Ura
        return @"g";
    if ( [planet hasPrefix:@"Nep"] )        // Nep
        return @"h";
    if ( [planet hasPrefix:@"Plu"] )        // Plu
        return @"i";
    if ( [planet hasPrefix:@"Chi"] )        // Chi
        return @"j";
    if ( [planet hasPrefix:@"Moo"] )        // Moo
        return @"k";
    if ( [planet isEqual:@"Lilith"] || [planet hasPrefix:@"Apo"] || [planet isEqual:@"True Apogee"] )
        return @"l";
    if ( [planet hasPrefix:@"Nod"]  || [planet isEqual:@"North Node"] || [planet isEqual:@"True Node"] )
        return @"m";
    if ( [planet isEqual:@"South Node"] )
        return @"n";
    if ( [planet isEqual:@"Ceres"] )
        return @"Y";
    if ( [planet isEqual:@"Juno"] || [planet isEqual:@"Hera"] )
        return @"Z";
    if ( [planet isEqual:@"Pallas"] || [planet isEqual:@"Minerva"] )
        return @"[";
    if ( [planet isEqual:@"Vesta"] )
        return @"\\";
    if ( [planet isEqual:@"Pholus"] )
        return @"o";
    if ( [planet isEqual:@"AC"] )
        return @"P";
    if ( [planet isEqual:@"DC"] )
        return @"Q";
    if ( [planet isEqual:@"MC"] )
        return @"R";
    if ( [planet isEqual:@"IC"] )
        return @"S";
    if ( [planet isEqual:@"R"] )
        return @"\x60";
    if ( [planet isEqual:@"r"] )	// index R
        return @"X";
    if ( [planet isEqual:@"North"] )
        return @"T";
    if ( [planet isEqual:@"South"] )
        return @"U";
    if ( [planet isEqual:@"Apogee"] )
        return @"V";
    if ( [planet isEqual:@"Peregee"] )
        return @"W";
    return @"?";
}

/*NSString *nodeSymbol(NSString *planet)
{
    if ( [planet isEqual:@"Mercury"] )
        return @"R";
    if ( [planet isEqual:@"Venus"] )
        return @"S";
    if ( [planet isEqual:@"Mars"] )
        return @"T";
    if ( [planet isEqual:@"Jupiter"] )
        return @"U";
    if ( [planet isEqual:@"Saturn"] )
        return @"V";
    if ( [planet isEqual:@"Uranus"] )
        return @"W";
    if ( [planet isEqual:@"Neptune"] )
        return @"X";
    if ( [planet isEqual:@"Pluto"] )
        return @"Y";
    if ( [planet isEqual:@"Chiron"] )
        return @"Z";
    if ( [planet isEqual:@"Moon"] )
        return @"m";
    return @"?";
}*/
NSString *nodeSymbol(NSString *planet)
{
    if ( [planet isEqual:@"Sun"] )
        return @"aT";
    if ( [planet isEqual:@"Mercury"] )
        return @"bT";
    if ( [planet isEqual:@"Venus"] )
        return @"cT";
    if ( [planet isEqual:@"Mars"] )
        return @"dT";
    if ( [planet isEqual:@"Jupiter"] )
        return @"eT";
    if ( [planet isEqual:@"Saturn"] )
        return @"fT";
    if ( [planet isEqual:@"Uranus"] )
        return @"gT";
    if ( [planet isEqual:@"Neptune"] )
        return @"hT";
    if ( [planet isEqual:@"Pluto"] )
        return @"iT";
    if ( [planet isEqual:@"Chiron"] )
        return @"jT";
    if ( [planet isEqual:@"Moon"] )
        return @"kT";
    if ( [planet isEqual:@"Ceres"] )
        return @"YT";
    if ( [planet isEqual:@"Juno"] || [planet isEqual:@"Hera"] )
        return @"ZT";
    if ( [planet isEqual:@"Pallas"] || [planet isEqual:@"Minerva"] )
        return @"[T";
    if ( [planet isEqual:@"Vesta"] )
        return @"\\T";
    if ( [planet isEqual:@"Pholus"] )
        return @"oT";
    return @"?";
}

NSString *southNodeSymbol(NSString *planet)
{
    if ( [planet isEqual:@"Sun"] )
        return @"aU";
    if ( [planet isEqual:@"Mercury"] )
        return @"bU";
    if ( [planet isEqual:@"Venus"] )
        return @"cU";
    if ( [planet isEqual:@"Mars"] )
        return @"dU";
    if ( [planet isEqual:@"Jupiter"] )
        return @"eU";
    if ( [planet isEqual:@"Saturn"] )
        return @"fU";
    if ( [planet isEqual:@"Uranus"] )
        return @"gU";
    if ( [planet isEqual:@"Neptune"] )
        return @"hU";
    if ( [planet isEqual:@"Pluto"] )
        return @"iU";
    if ( [planet isEqual:@"Chiron"] )
        return @"jU";
    if ( [planet isEqual:@"Moon"] )
        return @"kU";
    if ( [planet isEqual:@"Ceres"] )
        return @"YU";
    if ( [planet isEqual:@"Juno"] || [planet isEqual:@"Hera"] )
        return @"ZU";
    if ( [planet isEqual:@"Pallas"] || [planet isEqual:@"Minerva"] )
        return @"[U";
    if ( [planet isEqual:@"Vesta"] )
        return @"\\U";
    if ( [planet isEqual:@"Pholus"] )
        return @"oU";
    return @"?";
}

/* sign = 0 - 11
 */
NSString *zodiacSign(int sign)
{
    switch (sign)
    {
        case 0:		// Aries !
            return @"A";
        case 1:		// Taurus
            return @"B";
        case 2:		// Gemini
            return @"C";
        case 3:		// Cancer !
            return @"D";
        case 4:		// Leo
            return @"E";
        case 5:		// Virgo
            return @"F";
        case 6:		// Libra
            return @"G";
        case 7:		// Scorpio
            return @"H";
        case 8:		// Sagittarius
            return @"I";
        case 9:		// Capricorn
            return @"J";
        case 10:	// Aquarius
            return @"K";
        case 11:	// Pisces
            return @"L";
    }
    return @"?";
}

/* aspect symbol
 * deg = [0, 360]
 * orb = [0, 360]
 */
NSString *aspectSymbol(float deg, float orb)
{
    if (Diff(deg,   0.0) <= orb || Diff(deg, 360.0) <= orb)	// 0, 360   -> conjunct
        return @"p";
    if (Diff(deg,  30.0) <= orb || Diff(deg, 330.0) <= orb)	// 30, 330
        return @"q";
    if (Diff(deg,  45.0) <= orb || Diff(deg, 315.0) <= orb)	// 45, 315  -> semi square
        return @"r";
    if (Diff(deg,  60.0) <= orb || Diff(deg, 300.0) <= orb)	// 60, 300  -> sextile
        return @"s";
    if (Diff(deg,  72.0) <= orb || Diff(deg, 288.0) <= orb)	// 72, 288  -> septil
        return @"t";
    if (Diff(deg,  90.0) <= orb || Diff(deg, 270.0) <= orb)	// 90, 270  -> square
        return @"u";
    if (Diff(deg, 120.0) <= orb || Diff(deg, 240.0) <= orb)	// 120, 240 -> trigon
        return @"v";
    if (Diff(deg, 135.0) <= orb || Diff(deg, 225.0) <= orb)	// 135, 225 -> semi square
        return @"w";
    if (Diff(deg, 144.0) <= orb || Diff(deg, 216.0) <= orb)	// 144, 216 -> septil
        return @"x";
    if (Diff(deg, 150.0) <= orb || Diff(deg, 210.0) <= orb)	// 150, 210
        return @"y";
    if (Diff(deg, 180.0) <= orb)				// 180 -> opposition
        return @"z";
    return nil;
}

/* remainder = num % denom
 */
/*double modulo(double num, double denom)
{
    return (num - floor(num / denom) * denom);
}*/

/* bring deg to interval [0, 360]
 */
double mod360(double deg)
{
    while ( deg >= 360.0 )
        deg -= 360.0;
    while ( deg < 0.0 )
        deg += 360.0;
    return deg;
}
/* put angle within [0, 359.999] = modulo(360)
 */
double standardAngle( double angle )	// same as angle
{
    while ( angle >= 360.0 )
        angle -= 360.0;
    while ( angle < 0.0 )
        angle += 360.0;
    return angle;
}

/* angle = object1 - object2
 */
double angle180( double deg1, double deg2 )
{   double ang = angle(deg1, deg2);

    if ( ang >= 180.0 )
        ang = 360.0 - ang;
    return ang;
}
/* angle = modulo(object1 - object2, 360)
 */
double angle( double deg1, double deg2 )
{   double angle = deg1 - deg2;

    while ( angle >= 360.0 )
        angle -= 360.0;
    while ( angle < 0.0 )
        angle += 360.0;
    return angle;
}
/* mirror = object1 - (360 - object2)
 */
double mirror( double deg1, double deg2 )
{
    return angle( deg1, 360.0 - deg2 );
}

/* midpoint between object1 and object2
 * created: 2005-03-15
 */
double midpoint(double deg1, double deg2)
{
    if (Diff(deg1, deg2) > 180.0)	// flip over 0/360
    {
        if (deg1 < deg2) deg1 += 360.0;
        else             deg2 += 360.0;
    }
    return (deg1 + deg2) / 2.0;
}

/* harmony = 180 * (obj - obj1) / (obj2 - obj1)
 * obj1 represents 0 degree, obj2 represents 180 degree
 * we return the angle obj represents between obj1 and obj2
 */
double harmony( double obj, double obj1, double obj2)
{   double	a, b;

    a = angle(obj2, obj1);
    b = angle(obj, obj1);
    if (b < a && a >= 0.0001)
        return 180.0 * b / a;
    return -180.0 * (180.0 - b) / (360.0 - a);
}


/* check degFr <= deg <= degTo -> return YES
 * we use a fixed modulus 360
 * and pay attention to overflowing degrees at the 0/360 degree boundary
 * parameter:
 *   degFr  degree at earlier time
 *   deg    degree to test
 *   degTo  degree at later time
 *   retro  NO  = direct     -> degFr < deg < degTo
 *          YES = retrograde -> degTo < deg < degFr
 */
BOOL vhfIsDegreeInsideRange(double deg, double degFr, double degTo, BOOL retro)
{
    if (retro)	// exchange degFr and degTo
    {   double  v = degFr;
        degFr = degTo;
        degTo = v;
    }
    if (degFr < 0.0)	// bring degFr above 0 (2006-05-31)
        degFr += 360.0;
    if (degTo < degFr)	// bring degTo behind degFr
        degTo += 360.0;
    if (deg < degFr)	// bring deg behind degFr
        deg += 360.0;
    if ( degFr <= deg && deg <= degTo)
        return YES;
    return NO;
}
